import { CanActivate, ExecutionContext, Injectable, UnauthorizedException, Logger } from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { UsersService } from '../users.service';

@Injectable()
export class AdminGuard implements CanActivate {
  private readonly logger = new Logger(AdminGuard.name);

  constructor(
    private jwtService: JwtService,
    private usersService: UsersService,
  ) {}


async canActivate(context: ExecutionContext): Promise<boolean> {
  const request = context.switchToHttp().getRequest();
  const token = this.extractTokenFromHeader(request);
  
  this.logger.debug('Authorization header:', request.headers.authorization);
  this.logger.debug('Extracted token:', token);
  
  if (!token) {
    throw new UnauthorizedException('No token provided');
  }

  try {
    const payload = await this.jwtService.verifyAsync(token);
    this.logger.debug('Token payload:', payload);
    
    const user = await this.usersService.findOne(payload.sub); // Fix: Use payload.sub
    this.logger.debug('Found user:', user);
    
    if (!user) {
      throw new UnauthorizedException('User not found');
    }
    
    // Check if user role is Admin - handle both string and object formats
    const userRole = typeof user.role === 'string' ? user.role : user.role?.name;
    if (userRole !== 'Admin') {
      throw new UnauthorizedException('Only Admin users can perform this action');
    }
    console.log('User Role', user.role);
    
    return true;
  } catch (error) {
    this.logger.error('Error in AdminGuard:', error);
    if (error instanceof UnauthorizedException) {
      throw error;
    }
    throw new UnauthorizedException('Invalid token or insufficient permissions');
  }
}

  private extractTokenFromHeader(request: any): string | undefined {
    const [type, token] = request.headers.authorization?.split(' ') ?? [];
    return type === 'Bearer' ? token : undefined;
  }
} 