import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Permission } from './entities/permission.entity';
import { PermissionsResponse, PermissionResponse } from './permissions.model';

@Injectable()
export class PermissionsService {
  constructor(
    @InjectRepository(Permission)
    private permissionsRepository: Repository<Permission>,
  ) {}

  private getContentType(entity: string): number {
    // Map entity names to their corresponding content_type values
    const contentTypeMap: { [key: string]: number } = {
      'log entry': 1,
      'permission': 2,
      'group': 3,
      'content type': 4,
      'session': 5,
      'blacklisted token': 6,
      'outstanding token': 7,
      'app version': 8,
      'audit log': 9,
      'otp': 10,
      'user activity': 11,
      'user notification': 12,
      'user': 13,
      'employees': 14,
      'historical employees': 15,
      'historical user': 16,
      'address': 23,
      'customer assignment history': 26,
      'customer notes': 27,
      'historical customer': 24,
      'customer': 25,
      'variation types': 28,
      'variation option': 29,
      'product': 30,
      'product variation': 31,
      'variation details': 32,
      'stock update log': 33,
      'order details': 34,
      'order item': 35,
      'historical proforma': 36,
      'proforma': 37
    };

    // Check for exact match first
    if (contentTypeMap[entity.toLowerCase()]) {
      return contentTypeMap[entity.toLowerCase()];
    }

    // Check for alternative matches
    const altKey = `${entity.toLowerCase()}_alt`;
    if (contentTypeMap[altKey]) {
      return contentTypeMap[altKey];
    }

    return 0;
  }

  private formatCodename(action: string, entity: string): string {
    // Special cases for codename formatting
    if (action.includes('Assigned To')) {
      return 'can_change_assigned_to';
    } else if (action.includes('DNC')) {
      return 'can_change_dnc';
    } else if (action.includes('Stock Level')) {
      return 'can_change_stock_level';
    } else if (action.includes('confirm order status')) {
      return 'can_confirm_order';
    }

    // Extract the action part from the permission name
    const actionMatch = action.match(/Can\s+(\w+)\s+(.+)/);
    if (!actionMatch) return action.toLowerCase().replace(/\s+/g, '_');

    const actionPart = actionMatch[1].toLowerCase();
    const entityPart = actionMatch[2].toLowerCase().replace(/\s+/g, '');

    return `${actionPart}_${entityPart}`;
  }

  async findAll(role?: string): Promise<PermissionsResponse & { customer_permissions: PermissionResponse[], user_permissions: PermissionResponse[] }> {
    let permissions: Permission[];
    
    if (!role) {
      permissions = await this.permissionsRepository.find({
        order: {
          category: 'ASC',
          entity: 'ASC',
          action: 'ASC',
        },
      });
    } else {
      // Map role input to role_id
      let roleId: string;
      switch (role.toLowerCase()) {
        case '1':
        case 'admin':
          roleId = '1';
          break;
        case '2':
        case 'retention':
          roleId = '2';
          break;
        case '3':
        case 'reactivation':
          roleId = '3';
          break;
        case '4':
        case 'owner':
          roleId = '4';
          break;
        case '5':
        case 'escalator':
          roleId = '5';
          break;
        case '6':
        case 'agent':
          roleId = '6';
          break;
        case '7':
        case 'team lead':
          roleId = '7';
          break;
        default:
          return {
            data: [],
            customer_permissions: [],
            user_permissions: [],
            success: true,
            message: 'List Successfully Retrieved!',
            status: 200
          };
      }

      // Admin gets all permissions
      if (roleId === '1' || roleId === '4') {
        permissions = await this.permissionsRepository.find({
          order: {
            category: 'ASC',
            entity: 'ASC',
            action: 'ASC',
          },
        });
      } else {
        // Find permissions where role_id is in the comma-separated role column
        permissions = await this.permissionsRepository
          .createQueryBuilder('permission')
          .where('permission.role LIKE :rolePattern', { rolePattern: `%${roleId}%` })
          .orderBy('permission.category', 'ASC')
          .addOrderBy('permission.entity', 'ASC')
          .addOrderBy('permission.action', 'ASC')
          .getMany();
      }
    }

    const formattedPermissions: PermissionResponse[] = permissions.map(permission => ({
      id: permission.id,
      name: permission.action,
      codename: this.formatCodename(permission.action, permission.entity),
      content_type: this.getContentType(permission.entity)
    }));

    // Filter for customer and user permissions
    const customerContentTypes = [11, 23, 15, 25, 16, 27];
    const userContentTypes = [8, 14];
    const customer_permissions = formattedPermissions.filter(p => customerContentTypes.includes(p.content_type));
    const user_permissions = formattedPermissions.filter(p => userContentTypes.includes(p.content_type));
    // Remove these from the main data array
    const groupedIds = new Set([
      ...customer_permissions.map(p => p.id),
      ...user_permissions.map(p => p.id)
    ]);
    const filteredData = formattedPermissions.filter(p => !groupedIds.has(p.id));

    // Deduplicate customer_permissions by codename only
    const uniqueCustomerPermissions = Array.from(
      new Map(customer_permissions.map(p => [p.codename, p])).values()
    );
    // Deduplicate user_permissions by codename only
    const uniqueUserPermissions = Array.from(
      new Map(user_permissions.map(p => [p.codename, p])).values()
    );

    // Sort uniqueCustomerPermissions according to the required order
    const customerPermissionOrder = [
      { entity: 'customer', action: 'Can add customer' },
      { entity: 'customer', action: 'Can change customer' },
      { entity: 'customer', action: 'Can view customer' },
      { entity: 'customer notes', action: 'Can add customer notes' },
      { entity: 'customer notes', action: 'Can view customer notes' },
      { entity: 'address', action: 'Can add address' },
      { entity: 'address', action: 'Can change address' },
      { entity: 'address', action: 'Can view address' },
    ];
    const ordered = customerPermissionOrder
      .map(order => uniqueCustomerPermissions.find(p =>
        p.name === order.action && p.codename.includes(order.entity.replace(/\s/g, ''))
      ))
      .filter((p): p is PermissionResponse => Boolean(p));
    // Add the rest of the uniqueCustomerPermissions that are not in the ordered list
    const orderedIds = new Set(ordered.map(p => p.id));
    const rest = uniqueCustomerPermissions.filter(p => !orderedIds.has(p.id));
    const sortedCustomerPermissions = [...ordered, ...rest];

    // Only include these entities for customer_permissions
    const allowedCustomerEntities = ['address', 'customer', 'customer notes'];
    const customerCategoryPermissions = permissions.filter(
      p => p.category === 'Customers' && allowedCustomerEntities.includes(p.entity.toLowerCase())
    );
    const formattedCustomerCategoryPermissions: PermissionResponse[] = customerCategoryPermissions.map(permission => ({
      id: permission.id,
      name: permission.action,
      codename: this.formatCodename(permission.action, permission.entity),
      content_type: this.getContentType(permission.entity)
    }));
    // Deduplicate by codename
    const uniqueCustomerCategoryPermissions = Array.from(
      new Map(formattedCustomerCategoryPermissions.map(p => [p.codename, p])).values()
    );
    // Sort according to the required order
    const orderedCustomerPermissions = customerPermissionOrder
      .map(order => uniqueCustomerCategoryPermissions.find(p =>
        p.name === order.action && p.codename.includes(order.entity.replace(/\s/g, ''))
      ))
      .filter((p): p is PermissionResponse => Boolean(p));
    const orderedCustomerIds = new Set(orderedCustomerPermissions.map(p => p.id));
    const restCustomerPermissions = uniqueCustomerCategoryPermissions.filter(p => !orderedCustomerIds.has(p.id));
    const sortedCustomerPermissionsCustomer = [...orderedCustomerPermissions, ...restCustomerPermissions];

    return {
      data: [], // Intentionally empty, only customer_permissions and user_permissions are shown
      customer_permissions: sortedCustomerPermissionsCustomer,
      user_permissions: uniqueUserPermissions,
      success: true,
      message: '',
      status: 200
    };
  }

  async findAllGrouped(role?: string) {
    const flat = await this.findAll(role);
    return {
      ...flat,
      grouped: this.groupPermissions(flat.data)
    };
  }

  async findByCategory(category: string): Promise<PermissionsResponse> {
    const permissions = await this.permissionsRepository.find({
      where: { category },
      order: {
        entity: 'ASC',
        action: 'ASC',
      },
    });

    const formattedPermissions: PermissionResponse[] = permissions.map(permission => ({
      id: permission.id,
      name: permission.action,
      codename: this.formatCodename(permission.action, permission.entity),
      content_type: this.getContentType(permission.entity)
    }));

    return {
      data: formattedPermissions,
      success: true,
      message: 'List Successfully Retrieved!',
      status: 200
    };
  }

  async findByEntity(category: string, entity: string): Promise<PermissionsResponse> {
    const permissions = await this.permissionsRepository.find({
      where: { category, entity },
      order: {
        action: 'ASC',
      },
    });

    const formattedPermissions: PermissionResponse[] = permissions.map(permission => ({
      id: permission.id,
      name: permission.action,
      codename: this.formatCodename(permission.action, permission.entity),
      content_type: this.getContentType(permission.entity)
    }));

    return {
      data: formattedPermissions,
      success: true,
      message: 'List Successfully Retrieved!',
      status: 200
    };
  }

  async findOne(id: number): Promise<Permission | null> {
    return this.permissionsRepository.findOne({
      where: { id },
    });
  }

  async getAllPermissions(): Promise<PermissionResponse[]> {
    const permissions = await this.permissionsRepository.find({
      order: {
        category: 'ASC',
        entity: 'ASC',
        action: 'ASC',
      },
    });

    return permissions.map(permission => ({
      id: permission.id,
      name: permission.action,
      codename: this.formatCodename(permission.action, permission.entity),
      content_type: this.getContentType(permission.entity)
    }));
  }

  public groupPermissions(permissions: PermissionResponse[]) {
    // Group permissions by module and submodule using PERMISSION_GROUPS
    const PERMISSION_GROUPS = {
      Customers: {
        address: [11, 23],
        customer: [15, 25],
        'customer notes': [16, 27],
      },
      Users: {
        employees: [8, 14],
      },
    };
    const grouped: any = {};
    for (const [module, submodules] of Object.entries(PERMISSION_GROUPS)) {
      grouped[module] = {};
      for (const [submodule, contentTypes] of Object.entries(submodules)) {
        grouped[module][submodule] = permissions.filter(p =>
          contentTypes.includes(p.content_type)
        );
      }
    }
    return grouped;
  }
}