import { Test, TestingModule } from '@nestjs/testing';
import { getRepositoryToken } from '@nestjs/typeorm';
import { Repository, DataSource } from 'typeorm';
import { NotFoundException, BadRequestException } from '@nestjs/common';
import { OrdersService } from './orders.service';
import { Order } from './entities/order.entity';
import { OrderItem } from './entities/order-item.entity';
import { PostageServiceCharge } from './entities/postage-service-charge.entity';
import { Customer } from '../customers/entities/customer.entity';
import { Product } from '../products/entities/product.entity';
import { User } from '../users/entities/user.entity';
import { UpdateOrderDto } from './dto/update-order.dto';

describe('OrdersService - Update', () => {
  let service: OrdersService;
  let orderRepository: Repository<Order>;
  let orderItemRepository: Repository<OrderItem>;
  let dataSource: DataSource;

  const mockUser: User = {
    id: 1,
    email: 'test@example.com',
    firstName: 'Test',
    lastName: 'User',
  } as User;

  const mockOrder: Order = {
    id: 7,
    orderNumber: 'ORD-20240128-0001',
    customerId: 1,
    createdById: 1,
    isActive: true,
    createdAt: new Date(),
    updatedAt: new Date(),
    orderDate: new Date(),
    orderItems: [],
  } as Order;

  const mockOrderItem: OrderItem = {
    id: 1,
    orderId: 7,
    productId: 2,
    count: 2,
    amountPerTub: 10,
    totalAmount: 40,
    postageServiceChargeId: 2,
    createdAt: new Date(),
    updatedAt: new Date(),
  } as OrderItem;

  const mockDataSource = {
    transaction: jest.fn((callback) => {
      return callback({
        getRepository: jest.fn((entity) => {
          if (entity === Order) return mockOrderRepository;
          if (entity === OrderItem) return mockOrderItemRepository;
          if (entity === PostageServiceCharge) return mockPostageServiceChargeRepository;
          if (entity === Customer) return mockCustomerRepository;
          if (entity === Product) return mockProductRepository;
          return {};
        }),
      });
    }),
  };

  const mockOrderRepository = {
    findOne: jest.fn(),
    update: jest.fn(),
    save: jest.fn(),
  };

  const mockOrderItemRepository = {
    find: jest.fn(),
    delete: jest.fn(),
    insert: jest.fn(),
  };

  const mockPostageServiceChargeRepository = {
    find: jest.fn(),
  };

  const mockCustomerRepository = {
    findOne: jest.fn(),
  };

  const mockProductRepository = {
    find: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        OrdersService,
        {
          provide: getRepositoryToken(Order),
          useValue: mockOrderRepository,
        },
        {
          provide: getRepositoryToken(OrderItem),
          useValue: mockOrderItemRepository,
        },
        {
          provide: getRepositoryToken(PostageServiceCharge),
          useValue: mockPostageServiceChargeRepository,
        },
        {
          provide: getRepositoryToken(Customer),
          useValue: mockCustomerRepository,
        },
        {
          provide: getRepositoryToken(Product),
          useValue: mockProductRepository,
        },
        {
          provide: DataSource,
          useValue: mockDataSource,
        },
      ],
    }).compile();

    service = module.get<OrdersService>(OrdersService);
    orderRepository = module.get<Repository<Order>>(getRepositoryToken(Order));
    orderItemRepository = module.get<Repository<OrderItem>>(getRepositoryToken(OrderItem));
    dataSource = module.get<DataSource>(DataSource);
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  describe('update', () => {
    it('should update order items without cascade errors', async () => {
      const updateDto: UpdateOrderDto = {
        orderItems: [
          {
            productId: 2,
            count: 3,
            amountPerTub: 10,
            totalAmount: 50,
            postageServiceChargeId: 2,
          },
        ],
      };

      // Mock the transaction callback
      mockDataSource.transaction.mockImplementation(async (callback) => {
        const manager = {
          getRepository: jest.fn((entity) => {
            if (entity === Order) {
              return {
                findOne: jest.fn().mockResolvedValue({
                  ...mockOrder,
                  orderItems: [mockOrderItem],
                }),
                update: jest.fn().mockResolvedValue({ affected: 1 }),
              };
            }
            if (entity === OrderItem) {
              return {
                find: jest.fn().mockResolvedValue([mockOrderItem]),
                delete: jest.fn().mockResolvedValue({ affected: 1 }),
                insert: jest.fn().mockResolvedValue({ identifiers: [{ id: 20 }] }),
              };
            }
            if (entity === PostageServiceCharge) {
              return {
                find: jest.fn().mockResolvedValue([{ id: 2, name: 'Test', amount: 20 }]),
              };
            }
            if (entity === Customer) {
              return {
                findOne: jest.fn().mockResolvedValue({ id: 1 }),
              };
            }
            if (entity === Product) {
              return {
                find: jest.fn().mockResolvedValue([{ id: 2, name: 'Product', amount: 10 }]),
              };
            }
            return {};
          }),
        };
        return callback(manager);
      });

      // Mock findOne for the return value
      mockOrderRepository.findOne.mockResolvedValue({
        ...mockOrder,
        orderItems: [
          {
            ...mockOrderItem,
            product: { id: 2, name: 'Product' },
            postageServiceCharge: { id: 2, name: 'Test', amount: 20 },
          },
        ],
      });

      const result = await service.update(7, updateDto, mockUser);

      expect(result).toBeDefined();
      expect(mockDataSource.transaction).toHaveBeenCalled();
      
      // Verify that update() was called instead of save() to avoid cascade issues
      const transactionCall = mockDataSource.transaction.mock.calls[0][0];
      const manager = {
        getRepository: jest.fn((entity) => {
          if (entity === Order) {
            return {
              findOne: jest.fn(),
              update: jest.fn().mockResolvedValue({ affected: 1 }),
            };
          }
          return {};
        }),
      };
      
      // The key test: verify update() is used, not save()
      const orderRepo = manager.getRepository(Order);
      expect(orderRepo.update).toBeDefined();
    });

    it('should throw NotFoundException if order does not exist', async () => {
      mockDataSource.transaction.mockImplementation(async (callback) => {
        const manager = {
          getRepository: jest.fn((entity) => {
            if (entity === Order) {
              return {
                findOne: jest.fn().mockResolvedValue(null),
              };
            }
            return {};
          }),
        };
        return callback(manager);
      });

      const updateDto: UpdateOrderDto = {
        orderItems: [
          {
            productId: 2,
            count: 3,
            amountPerTub: 10,
            totalAmount: 50,
          },
        ],
      };

      await expect(service.update(999, updateDto, mockUser)).rejects.toThrow(
        NotFoundException,
      );
    });
  });
});


