import {
  Injectable,
  NotFoundException,
  BadRequestException,
  ConflictException,
  Logger,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, Between, Like, DataSource } from 'typeorm';
import { Order } from './entities/order.entity';
import { OrderItem } from './entities/order-item.entity';
import { PostageServiceCharge } from './entities/postage-service-charge.entity';
import { Customer } from '../customers/entities/customer.entity';
import { Product } from '../products/entities/product.entity';
import { User } from '../users/entities/user.entity';
import { CreateOrderDto } from './dto/create-order.dto';
import { UpdateOrderDto } from './dto/update-order.dto';
import { ListOrdersDto } from './dto/list-orders.dto';

@Injectable()
export class OrdersService {
  private readonly logger = new Logger(OrdersService.name);

  constructor(
    @InjectRepository(Order)
    private readonly ordersRepository: Repository<Order>,
    @InjectRepository(OrderItem)
    private readonly orderItemsRepository: Repository<OrderItem>,
    @InjectRepository(PostageServiceCharge)
    private readonly postageServiceChargesRepository: Repository<PostageServiceCharge>,
    @InjectRepository(Customer)
    private readonly customersRepository: Repository<Customer>,
    @InjectRepository(Product)
    private readonly productsRepository: Repository<Product>,
    @InjectRepository(User)
    private readonly usersRepository: Repository<User>,
    private readonly dataSource: DataSource,
  ) {}

  private async generateOrderNumber(): Promise<string> {
    const today = new Date();
    const dateStr = today.toISOString().split('T')[0].replace(/-/g, '');
    
    // Find the last order number for today (including null/empty ones)
    const lastOrder = await this.ordersRepository
      .createQueryBuilder('order')
      .where('order.orderNumber LIKE :pattern', { pattern: `ORD-${dateStr}-%` })
      .orderBy('order.createdAt', 'DESC')
      .getOne();

    let sequence = 1;
    if (lastOrder && lastOrder.orderNumber) {
      const parts = lastOrder.orderNumber.split('-');
      if (parts.length >= 3) {
        const lastSequence = parseInt(parts[2] || '0', 10);
        sequence = lastSequence + 1;
      }
    }

    const orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
    
    // Double-check uniqueness
    const exists = await this.ordersRepository.findOne({
      where: { orderNumber },
    });
    
    if (exists) {
      // If somehow exists, increment and try again
      sequence++;
      return `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
    }

    return orderNumber;
  }

  async cleanupInvalidOrders(): Promise<void> {
    try {
      // First, fix empty order numbers
      await this.fixEmptyOrderNumbers();
      
      // Then, remove orders with invalid customerIds
      const allOrders = await this.ordersRepository.find();
      const allCustomerIds = (await this.customersRepository.find()).map(c => c.id);
      
      const invalidOrders = allOrders.filter(order => 
        order.customerId !== null && order.customerId !== undefined && !allCustomerIds.includes(order.customerId)
      );
      
      if (invalidOrders.length > 0) {
        console.log(`Found ${invalidOrders.length} orders with invalid customerIds. Removing...`);
        
        // Delete order items first (cascade should handle this, but being explicit)
        for (const order of invalidOrders) {
          await this.orderItemsRepository.delete({ orderId: order.id });
        }
        
        // Then delete the orders
        await this.ordersRepository.remove(invalidOrders);
        console.log(`Removed ${invalidOrders.length} orders with invalid customerIds.`);
      }
    } catch (error) {
      console.warn('Failed to cleanup invalid orders:', error.message);
    }
  }

  async fixEmptyOrderNumbers(): Promise<void> {
    try {
      // Find all orders with empty or null orderNumber
      const ordersWithoutNumber = await this.ordersRepository
        .createQueryBuilder('order')
        .where('order.orderNumber IS NULL OR order.orderNumber = :empty', { empty: '' })
        .getMany();

      if (ordersWithoutNumber.length > 0) {
        console.log(`Found ${ordersWithoutNumber.length} orders without order numbers. Fixing...`);
        
        for (const order of ordersWithoutNumber) {
          // Generate order number based on creation date
          const orderDate = order.createdAt || new Date();
          const dateStr = orderDate.toISOString().split('T')[0].replace(/-/g, '');
          
          // Find the highest sequence for that date (including all orders, not just existing ones)
          const allOrders = await this.ordersRepository
            .createQueryBuilder('o')
            .where('o.orderNumber IS NOT NULL AND o.orderNumber != :empty', { empty: '' })
            .andWhere('o.orderNumber LIKE :pattern', { pattern: `ORD-${dateStr}-%` })
            .getMany();
          
          let maxSequence = 0;
          allOrders.forEach(o => {
            if (o.orderNumber) {
              const parts = o.orderNumber.split('-');
              if (parts.length >= 3) {
                const seq = parseInt(parts[2] || '0', 10);
                if (seq > maxSequence) maxSequence = seq;
              }
            }
          });
          
          // Generate unique order number
          let sequence = maxSequence + 1;
          let orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
          
          // Ensure uniqueness by checking and incrementing if needed
          let exists = await this.ordersRepository.findOne({
            where: { orderNumber },
          });
          
          while (exists) {
            sequence++;
            orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
            exists = await this.ordersRepository.findOne({
              where: { orderNumber },
            });
          }
          
          order.orderNumber = orderNumber;
          await this.ordersRepository.save(order);
        }
        
        console.log(`Fixed ${ordersWithoutNumber.length} orders with missing order numbers.`);
      }
    } catch (error) {
      console.warn('Failed to fix empty order numbers:', error.message);
    }
  }

  async create(createOrderDto: CreateOrderDto, user: User): Promise<Order> {
    try {
      this.logger.debug(`create() - Creating order for customer ID: ${createOrderDto.customerId}, user ID: ${user.id}, items count: ${createOrderDto.orderItems?.length || 0}`);
      
      // Validate customerId is provided
      if (!createOrderDto.customerId) {
        this.logger.warn(`create() - BadRequest: Customer ID is required`);
        throw new BadRequestException('Customer ID is required');
      }

      // Validate customer exists
      const customer = await this.customersRepository.findOne({
        where: { id: createOrderDto.customerId },
      });

      if (!customer) {
        this.logger.warn(`create() - NotFoundException: Customer with ID ${createOrderDto.customerId} not found`);
        throw new NotFoundException(`Customer with ID ${createOrderDto.customerId} not found`);
      }
      this.logger.debug(`create() - Customer validated: ${customer.id}`);

      // Validate all products exist
      const productIds = createOrderDto.orderItems.map((item) => item.productId);
      this.logger.debug(`create() - Validating ${productIds.length} products: ${productIds.join(', ')}`);
      const products = await this.productsRepository.find({
        where: productIds.map((id) => ({ id })),
      });

      if (products.length !== productIds.length) {
        const foundIds = products.map((p) => p.id);
        const missingIds = productIds.filter((id) => !foundIds.includes(id));
        this.logger.warn(`create() - NotFoundException: Products with IDs ${missingIds.join(', ')} not found`);
        throw new NotFoundException(
          `Products with IDs ${missingIds.join(', ')} not found`,
        );
      }
      this.logger.debug(`create() - All products validated`);

      // Validate postage service charges if provided
      const postageServiceChargeIds = createOrderDto.orderItems
        .map((item) => item.postageServiceChargeId)
        .filter((id) => id !== undefined && id !== null);
      
      if (postageServiceChargeIds.length > 0) {
        this.logger.debug(`create() - Validating ${postageServiceChargeIds.length} postage/service charges`);
        const uniqueIds = [...new Set(postageServiceChargeIds)];
        const charges = await this.postageServiceChargesRepository.find({
          where: uniqueIds.map((id) => ({ id })),
        });

        if (charges.length !== uniqueIds.length) {
          const foundIds = charges.map((c) => c.id);
          const missingIds = uniqueIds.filter((id) => !foundIds.includes(id));
          this.logger.warn(`create() - NotFoundException: Postage/Service charges with IDs ${missingIds.join(', ')} not found`);
          throw new NotFoundException(
            `Postage/Service charges with IDs ${missingIds.join(', ')} not found`,
          );
        }
        this.logger.debug(`create() - All postage/service charges validated`);
      }

      // Generate unique order number (required, cannot be null)
      this.logger.debug(`create() - Generating order number`);
      const orderNumber = await this.generateOrderNumber();
      if (!orderNumber) {
        this.logger.error(`create() - BadRequestException: Failed to generate order number`);
        throw new BadRequestException('Failed to generate order number');
      }
      this.logger.debug(`create() - Generated order number: ${orderNumber}`);

      // Parse order date or use current date
      const orderDate = createOrderDto.orderDate 
        ? new Date(createOrderDto.orderDate) 
        : new Date();

      // Create order
      this.logger.debug(`create() - Creating order entity`);
      const order = this.ordersRepository.create({
        orderNumber,
        customerId: createOrderDto.customerId,
        createdById: user.id,
        orderDate,
      });

      const savedOrder = await this.ordersRepository.save(order);
      this.logger.debug(`create() - Order saved with ID: ${savedOrder.id}`);

      // Create order items
      this.logger.debug(`create() - Creating ${createOrderDto.orderItems.length} order items`);
      const orderItems = createOrderDto.orderItems.map((item) => {
        const orderItem = new OrderItem();
        orderItem.orderId = savedOrder.id;
        orderItem.productId = item.productId;
        orderItem.count = item.count;
        orderItem.amountPerTub = item.amountPerTub;
        orderItem.totalAmount = item.totalAmount;
        if (item.postageServiceChargeId !== undefined && item.postageServiceChargeId !== null) {
          orderItem.postageServiceChargeId = item.postageServiceChargeId;
        }
        if (item.deductionNote !== undefined && item.deductionNote !== null) {
          orderItem.deductionNote = item.deductionNote;
        }
        return orderItem;
      });

      await this.orderItemsRepository.save(orderItems);
      this.logger.debug(`create() - Order items saved`);

      // Return order with relations
      const result = await this.findOne(savedOrder.id, user);
      this.logger.log(`create() - Successfully created order with ID: ${result.id}, orderNumber: ${result.orderNumber}`);
      return result;
    } catch (error) {
      const errorType = error.constructor.name;
      if (error instanceof NotFoundException || error instanceof BadRequestException) {
        this.logger.error(`create() - ${errorType}: ${error.message}`);
        throw error;
      }
      this.logger.error(`create() - Error creating order. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
      throw new BadRequestException(`Failed to create order: ${error.message || 'Unknown error occurred'}`);
    }
  }

  async findAll(
    listOrdersDto: ListOrdersDto,
    user: User,
  ): Promise<{ data: Order[]; total: number }> {
    try {
      const {
        page = 1,
        limit = 10,
        customerId,
        orderNumber,
        createdById,
        startDate,
        endDate,
      } = listOrdersDto;
      const skip = (page - 1) * limit;
      
      this.logger.debug(`findAll() - Finding orders with filters: page=${page}, limit=${limit}, customerId=${customerId}, orderNumber=${orderNumber}, createdById=${createdById}, startDate=${startDate}, endDate=${endDate}, user ID: ${user.id}`);

      const queryBuilder = this.ordersRepository
        .createQueryBuilder('order')
        .leftJoinAndSelect('order.customer', 'customer')
        .leftJoinAndSelect('order.createdBy', 'createdBy')
        .leftJoinAndSelect('order.orderItems', 'orderItems')
        .leftJoinAndSelect('orderItems.product', 'product')
        .leftJoinAndSelect('orderItems.postageServiceCharge', 'postageServiceCharge');

      if (customerId) {
        queryBuilder.where('order.customerId = :customerId', { customerId });
        this.logger.debug(`findAll() - Applied customerId filter: ${customerId}`);
      }

      if (orderNumber) {
        if (customerId) {
          queryBuilder.andWhere('order.orderNumber LIKE :orderNumber', {
            orderNumber: `%${orderNumber}%`,
          });
        } else {
          queryBuilder.where('order.orderNumber LIKE :orderNumber', {
            orderNumber: `%${orderNumber}%`,
          });
        }
        this.logger.debug(`findAll() - Applied orderNumber filter: ${orderNumber}`);
      }

      if (createdById) {
        const whereClause = customerId || orderNumber ? 'andWhere' : 'where';
        queryBuilder[whereClause]('order.createdById = :createdById', {
          createdById,
        });
        this.logger.debug(`findAll() - Applied createdById filter: ${createdById}`);
      }

      if (startDate && endDate) {
        const whereClause =
          customerId || orderNumber || createdById ? 'andWhere' : 'where';
        queryBuilder[whereClause]('order.createdAt BETWEEN :startDate AND :endDate', {
          startDate,
          endDate,
        });
        this.logger.debug(`findAll() - Applied date range filter: ${startDate} to ${endDate}`);
      } else if (startDate) {
        const whereClause =
          customerId || orderNumber || createdById ? 'andWhere' : 'where';
        queryBuilder[whereClause]('order.createdAt >= :startDate', { startDate });
        this.logger.debug(`findAll() - Applied startDate filter: ${startDate}`);
      } else if (endDate) {
        const whereClause =
          customerId || orderNumber || createdById ? 'andWhere' : 'where';
        queryBuilder[whereClause]('order.createdAt <= :endDate', { endDate });
        this.logger.debug(`findAll() - Applied endDate filter: ${endDate}`);
      }

      queryBuilder
        .andWhere('order.isActive = :isActive', { isActive: true })
        .skip(skip)
        .take(limit)
        .orderBy('order.createdAt', 'DESC');

      const [data, total] = await queryBuilder.getManyAndCount();
      
      this.logger.log(`findAll() - Successfully retrieved ${data.length} orders (total: ${total})`);
      return { data, total };
    } catch (error) {
      const errorType = error.constructor.name;
      this.logger.error(`findAll() - Error finding orders. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
      throw error;
    }
  }

  async findOne(id: number, user: User): Promise<Order> {
    try {
      this.logger.debug(`findOne() - Searching for order with ID: ${id}, user ID: ${user.id}`);
      
      const order = await this.ordersRepository.findOne({
        where: { id },
        relations: [
          'customer', 
          'createdBy', 
          'orderItems', 
          'orderItems.product',
          'orderItems.postageServiceCharge',
        ],
      });

      if (!order) {
        this.logger.warn(`findOne() - Order with ID ${id} not found`);
        throw new NotFoundException(`Order with ID ${id} not found`);
      }

      this.logger.log(`findOne() - Successfully found order with ID: ${id}, orderNumber: ${order.orderNumber}`);
      return order;
    } catch (error) {
      const errorType = error.constructor.name;
      if (error instanceof NotFoundException) {
        throw error;
      }
      this.logger.error(`findOne() - Error finding order with ID ${id}. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
      throw error;
    }
  }

  async update(
    id: number,
    updateOrderDto: UpdateOrderDto,
    user: User,
  ): Promise<Order> {
    try {
      this.logger.debug(`update() - Updating order with ID: ${id}, user ID: ${user.id}, data: ${JSON.stringify(updateOrderDto)}`);
      
      // Use transaction to ensure atomicity
      return await this.dataSource.transaction(async (manager) => {
      const orderRepository = manager.getRepository(Order);
      const orderItemsRepository = manager.getRepository(OrderItem);
      const postageServiceChargesRepository = manager.getRepository(PostageServiceCharge);
      const customersRepository = manager.getRepository(Customer);
      const productsRepository = manager.getRepository(Product);

      // Fetch order with existing order items to compare amounts
      // Don't load orderItems relation to avoid TypeORM tracking issues
      this.logger.debug(`update() - Fetching existing order with ID: ${id}`);
      const existingOrder = await orderRepository.findOne({
        where: { id },
        relations: ['customer', 'createdBy'],
      });

      if (!existingOrder) {
        this.logger.warn(`update() - NotFoundException: Order with ID ${id} not found`);
        throw new NotFoundException(`Order with ID ${id} not found`);
      }
      this.logger.debug(`update() - Found existing order: ${existingOrder.orderNumber}`);

      // Fetch order items separately for comparison
      const existingOrderItems = await orderItemsRepository.find({
        where: { orderId: id },
      });
      this.logger.debug(`update() - Found ${existingOrderItems.length} existing order items`);

      // Update customer if provided
      if (updateOrderDto.customerId !== undefined) {
        this.logger.debug(`update() - Validating customer ID: ${updateOrderDto.customerId}`);
        const customer = await customersRepository.findOne({
          where: { id: updateOrderDto.customerId },
        });

        if (!customer) {
          this.logger.warn(`update() - NotFoundException: Customer with ID ${updateOrderDto.customerId} not found`);
          throw new NotFoundException(
            `Customer with ID ${updateOrderDto.customerId} not found`,
          );
        }

        existingOrder.customerId = updateOrderDto.customerId;
        this.logger.debug(`update() - Customer updated to: ${updateOrderDto.customerId}`);
      }

      // Update order date if provided
      if (updateOrderDto.orderDate !== undefined) {
        existingOrder.orderDate = new Date(updateOrderDto.orderDate);
      }

      // Update order items if provided
      if (updateOrderDto.orderItems !== undefined) {
        this.logger.debug(`update() - Updating ${updateOrderDto.orderItems.length} order items`);
        
        // Validate all products exist
        const productIds = updateOrderDto.orderItems
          .map((item) => item.productId)
          .filter((id) => id !== undefined);
        
        if (productIds.length > 0) {
          this.logger.debug(`update() - Validating ${productIds.length} products: ${productIds.join(', ')}`);
          const products = await productsRepository.find({
            where: productIds.map((id) => ({ id })),
          });

          if (products.length !== productIds.length) {
            const foundIds = products.map((p) => p.id);
            const missingIds = productIds.filter((id) => !foundIds.includes(id));
            this.logger.warn(`update() - NotFoundException: Products with IDs ${missingIds.join(', ')} not found`);
            throw new NotFoundException(
              `Products with IDs ${missingIds.join(', ')} not found`,
            );
          }
          this.logger.debug(`update() - All products validated`);
        }

        // Validate postage service charges if provided
        const postageServiceChargeIds = updateOrderDto.orderItems
          .map((item) => item.postageServiceChargeId)
          .filter((id) => id !== undefined && id !== null);
        
        if (postageServiceChargeIds.length > 0) {
          this.logger.debug(`update() - Validating ${postageServiceChargeIds.length} postage/service charges`);
          const uniqueIds = [...new Set(postageServiceChargeIds)];
          const charges = await postageServiceChargesRepository.find({
            where: uniqueIds.map((id) => ({ id })),
          });

          if (charges.length !== uniqueIds.length) {
            const foundIds = charges.map((c) => c.id);
            const missingIds = uniqueIds.filter((id) => !foundIds.includes(id));
            this.logger.warn(`update() - NotFoundException: Postage/Service charges with IDs ${missingIds.join(', ')} not found`);
            throw new NotFoundException(
              `Postage/Service charges with IDs ${missingIds.join(', ')} not found`,
            );
          }
          this.logger.debug(`update() - All postage/service charges validated`);
        }

        // Check for amount decreases and validate deduction notes
        this.logger.debug(`update() - Checking for amount decreases and validating deduction notes`);
        for (const item of updateOrderDto.orderItems) {
          if (item.totalAmount !== undefined) {
            // Find existing order item to compare
            const existingItem = existingOrderItems.find(
              (oi) => oi.productId === item.productId,
            );

            if (existingItem && existingItem.totalAmount) {
              const oldAmount = parseFloat(existingItem.totalAmount.toString());
              const newAmount = parseFloat(item.totalAmount.toString());

              // If amount is decreased, require deduction note
              if (newAmount < oldAmount) {
                if (!item.deductionNote || item.deductionNote.trim() === '') {
                  this.logger.warn(`update() - BadRequestException: Deduction note required for product ID ${item.productId || existingItem.productId}, old amount: ${oldAmount}, new amount: ${newAmount}`);
                  throw new BadRequestException(
                    `Deduction note is required when decreasing total amount for product ID ${item.productId || existingItem.productId}. Old amount: ${oldAmount}, New amount: ${newAmount}`,
                  );
                }
                this.logger.debug(`update() - Deduction note provided for product ID ${item.productId || existingItem.productId}`);
              }
            }
          }
        }

        // Remove existing order items
        this.logger.debug(`update() - Removing existing order items`);
        await orderItemsRepository.delete({ orderId: id });

        // Create new order items - ensure all required fields are present
        this.logger.debug(`update() - Creating new order items`);
        const orderItems = updateOrderDto.orderItems.map((item) => {
          if (!item.productId) {
            this.logger.warn(`update() - BadRequestException: Product ID is required for order items`);
            throw new BadRequestException('Product ID is required for order items');
          }

          const orderItem = new OrderItem();
          orderItem.orderId = id;
          orderItem.productId = item.productId;
          orderItem.count = item.count !== undefined ? item.count : 1;
          
          // Set amountPerTub - required field
          if (item.amountPerTub !== undefined) {
            orderItem.amountPerTub = item.amountPerTub;
          } else {
            this.logger.warn(`update() - BadRequestException: amountPerTub is required for order items`);
            throw new BadRequestException('amountPerTub is required for order items');
          }
          
          // Set totalAmount - required field
          if (item.totalAmount !== undefined) {
            orderItem.totalAmount = item.totalAmount;
          } else {
            this.logger.warn(`update() - BadRequestException: totalAmount is required for order items`);
            throw new BadRequestException('totalAmount is required for order items');
          }
          
          // Set postageServiceChargeId if provided
          if (item.postageServiceChargeId !== undefined && item.postageServiceChargeId !== null) {
            orderItem.postageServiceChargeId = item.postageServiceChargeId;
          }
          
          // Set deductionNote if provided
          if (item.deductionNote !== undefined && item.deductionNote !== null) {
            orderItem.deductionNote = item.deductionNote;
          }
          
          return orderItem;
        });

        // Use insert instead of save to ensure new entities are created
        await orderItemsRepository.insert(orderItems.map(item => {
          const insertData: any = {
            orderId: item.orderId,
            productId: item.productId,
            count: item.count,
            amountPerTub: item.amountPerTub,
            totalAmount: item.totalAmount,
          };
          
          if (item.postageServiceChargeId !== undefined && item.postageServiceChargeId !== null) {
            insertData.postageServiceChargeId = item.postageServiceChargeId;
          }
          
          if (item.deductionNote !== undefined && item.deductionNote !== null) {
            insertData.deductionNote = item.deductionNote;
          }
          
          return insertData;
        }));
      }

      // Use update() instead of save() to avoid cascade issues with orderItems
      // This prevents TypeORM from trying to cascade-save orderItems relation
      const orderUpdateData: any = {};
      if (updateOrderDto.isActive !== undefined) {
        orderUpdateData.isActive = updateOrderDto.isActive;
      }
      if (updateOrderDto.customerId !== undefined) {
        orderUpdateData.customerId = updateOrderDto.customerId;
      }
      if (updateOrderDto.orderDate !== undefined) {
        orderUpdateData.orderDate = new Date(updateOrderDto.orderDate);
      }
      
      // Always update to ensure updated_at timestamp is refreshed
      // If no specific fields to update, just update the timestamp
      await orderRepository.update(id, orderUpdateData);

      // Return order with all relations
      const result = await this.findOne(id, user);
      this.logger.log(`update() - Successfully updated order with ID: ${id}, orderNumber: ${result.orderNumber}`);
      return result;
    });
    } catch (error) {
      const errorType = error.constructor.name;
      if (error instanceof NotFoundException || error instanceof BadRequestException) {
        this.logger.error(`update() - ${errorType}: ${error.message}`);
        throw error;
      }
      this.logger.error(`update() - Error updating order with ID ${id}. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
      throw new BadRequestException(`Failed to update order: ${error.message || 'Unknown error occurred'}`);
    }
  }

  async remove(id: number, user: User): Promise<void> {
    try {
      this.logger.debug(`remove() - Deleting order with ID: ${id}, user ID: ${user.id}`);
      
      const order = await this.findOne(id, user);
      // Soft delete
      order.isActive = false;
      await this.ordersRepository.save(order);
      
      this.logger.log(`remove() - Successfully deleted order with ID: ${id}, orderNumber: ${order.orderNumber}`);
    } catch (error) {
      const errorType = error.constructor.name;
      if (error instanceof NotFoundException) {
        throw error;
      }
      this.logger.error(`remove() - Error deleting order with ID ${id}. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
      throw error;
    }
  }
}

