import { Controller, Post, Get, Patch, Delete,Body, Query, HttpStatus, Param, UseGuards, BadRequestException, UploadedFile, UseInterceptors, Request} from '@nestjs/common';
import { CustomersService } from './customers.service';
import { CreateCustomerDto } from './dto/create-customer.dto';
import { UpdateCustomerDto } from './dto/update-customer.dto';
import { ListCustomersDto } from './dto/list-customers.dto';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { PermissionsGuard } from '../auth/guards/permissions.guard';
import { RequirePermission } from '../auth/decorators/require-permission.decorator';
import { User } from '../auth/decorators/user.decorator';
import { BulkAssignCustomerDto } from './dto/bulk-assign-customer.dto';
import { FileInterceptor } from '@nestjs/platform-express';
import { memoryStorage } from 'multer';
import { Roles } from '../auth/decorators/roles.decorator';
import { UserRole } from '../roles/roles.model';
import { RolesGuard } from '../auth/guards/roles.guard';

@Controller('api/customer/management')
@UseGuards(JwtAuthGuard, PermissionsGuard, RolesGuard)
export class CustomersController {
  constructor(private readonly customersService: CustomersService) {}

  @Post()
  // @RequirePermission('Can add customer')
  async create(@Body() createCustomerDto: CreateCustomerDto, @User() user: any) {
    const customer = await this.customersService.create(createCustomerDto, user);
    return {
      statusCode: HttpStatus.CREATED,
      message: 'Customer profile created successfully',
      data: {
        iso: customer.iso,
        title: customer.title,
        assigned_to_id: customer.assigned_to_id,
        last_name: customer.lastName,
        first_name: customer.firstName,
        email: customer.email,
        home_phone: customer.landline,
        mobile_phone: customer.mobileNumber,
        id: customer.id,
        gen_customer_id: customer.genCustomerId,
      },
    };
  }

 
  @Get(':id')
  // @RequirePermission('Can view customer')
  async findOne(@Param('id') id: number, @User() user: any) {
    try {
      const customer = await this.customersService.findOne(id, user);
      return {
        statusCode: HttpStatus.OK,
        message: 'Customer retrieved successfully',
        data: customer,
      };
    } catch (error) {
      console.error(`Error retrieving customer with id ${id} for user ${user?.id || 'unknown'}:`, error.message, error.stack);
      throw error;
    }
  }

  @Patch('bulk_assign')
  // @RequirePermission('Can change customer')
  async bulkAssignCustomers(@Body() dto: BulkAssignCustomerDto, @User() user: any) {
    const result = await this.customersService.bulkAssign(dto, user);
    return {
      statusCode: 200,
      success: true,
      message: {
        detail: 'Selected customers have been assigned',
      },
      data: result,
    };
  }

  @Patch(':id')
  // @RequirePermission('Can change customer')
  async update(@Param('id') id: string, @Body() updateCustomerDto: UpdateCustomerDto, @User() user: any) {
    try {
      const customer = await this.customersService.updateById(Number(id), updateCustomerDto, user);
      return {
        statusCode: HttpStatus.OK,
        message: 'Customer profile updated successfully',
        data: {
          iso: customer.iso,
          title: customer.title,
          assigned_to_id: customer.assigned_to_id,
          last_name: customer.lastName,
          first_name: customer.firstName,
          email: customer.email,
          home_phone: customer.landline,
          mobile_phone: customer.mobileNumber,
          id: customer.id,
          gen_customer_id: customer.genCustomerId,
        },
      };
    } catch (error) {
      console.error(`Error updating customer with id ${id}:`, error.message);
      throw error;
    }
  }

  @Get()
  // @RequirePermission('Can view customer')
  async findAll(@Query() listCustomersDto: ListCustomersDto, @User() user: any) {
    const { data, total } = await this.customersService.findAll(listCustomersDto, user);
    return {
      statusCode: HttpStatus.OK,
      message: 'Customers retrieved successfully',
      data,
      total,
    };
  }
@Post('bulk')
@Roles(UserRole.ADMIN, UserRole.OWNER)  // Only Admin can access this endpoint
@UseInterceptors(
  FileInterceptor('file', {
    storage: memoryStorage(), // Use memory storage to keep buffer available
    limits: {
      fileSize: 5 * 1024 * 1024, // 5MB limit
    },
    fileFilter: (req, file, callback) => {
      if (!file.originalname.match(/\.(xlsx|xls)$/)) {
        return callback(new BadRequestException('Only Excel files (.xlsx, .xls) are allowed!'), false);
      }
      callback(null, true);
    },
  }),
)
async bulkCreateCustomers(
  @UploadedFile() file: Express.Multer.File,
  @User() user: any,
) {
  // Debug logging
  console.log('File received:', file ? {
    originalname: file.originalname,
    mimetype: file.mimetype,
    size: file.size,
    bufferLength: file.buffer ? file.buffer.length : 0,
    hasBuffer: !!file.buffer
  } : 'FILE IS NULL/UNDEFINED');

  if (!file) {
    throw new BadRequestException('File is required');
  }

  if (!file.buffer || file.buffer.length === 0) {
    throw new BadRequestException('File is required and must not be empty. Please ensure the file is properly selected in Postman with type "File" (not "Text").');
  }

  const result = await this.customersService.bulkCreate(file, user);
  
  return {
    statusCode: HttpStatus.OK,
    success: true,
    message: result.message,
    data: result.details,
  };
}
}