import { TypeOrmModuleOptions } from '@nestjs/typeorm';
import { ConfigService } from '@nestjs/config';
import { DefaultNamingStrategy } from 'typeorm';
import * as mysql from 'mysql2/promise';

export const getDatabaseConfig = async (configService: ConfigService): Promise<TypeOrmModuleOptions> => {
  const config: TypeOrmModuleOptions = {
    type: 'mysql',
    host: configService.get('DB_HOST'),
    port: configService.get('DB_PORT'),
    username: configService.get('DB_USERNAME'),
    password: configService.get('DB_PASSWORD'),
    database: configService.get('DB_DATABASE'),
    entities: [__dirname + '/../**/*.entity{.ts,.js}'],
    synchronize: configService.get('DB_SYNCHRONIZE') === 'true', // Only enable if explicitly set to 'true'
    logging: configService.get('NODE_ENV') === 'development' ? ['query', 'error'] : ['error'], // Only log queries in development
    logger: 'advanced-console',
    namingStrategy: new DefaultNamingStrategy(),
  };

  // Clean up invalid orders before TypeORM syncs (if synchronize is enabled)
  if (configService.get('DB_SYNCHRONIZE') === 'true') {
    try {
      const connection = await mysql.createConnection({
        host: configService.get('DB_HOST'),
        port: configService.get('DB_PORT'),
        user: configService.get('DB_USERNAME'),
        password: configService.get('DB_PASSWORD'),
        database: configService.get('DB_DATABASE'),
      });

      // Delete order_items for orders with invalid customerIds
      // Using LEFT JOIN to find orders without matching customers
      await connection.execute(`
        DELETE oi FROM order_items oi
        INNER JOIN orders o ON oi.orderId = o.id
        LEFT JOIN customers c ON o.customerId = c.id
        WHERE o.customerId IS NOT NULL 
        AND c.id IS NULL
      `);

      // Delete orders with invalid customerIds
      await connection.execute(`
        DELETE o FROM orders o
        LEFT JOIN customers c ON o.customerId = c.id
        WHERE o.customerId IS NOT NULL 
        AND c.id IS NULL
      `);

      // Fix empty order numbers
      await connection.execute(`
        UPDATE orders 
        SET orderNumber = CONCAT('ORD-', DATE_FORMAT(created_at, '%Y%m%d'), '-', LPAD(id, 4, '0'))
        WHERE (orderNumber IS NULL OR orderNumber = '')
      `);

      await connection.end();
      console.log('Pre-sync cleanup completed: Invalid orders removed and empty order numbers fixed');
    } catch (error) {
      // If cleanup fails, log but don't crash - might be first run or table doesn't exist
      console.warn('Pre-sync cleanup skipped (this is normal on first run):', error.message);
    }
  }

  return config;
};