"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var OrdersService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrdersService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const order_entity_1 = require("./entities/order.entity");
const order_item_entity_1 = require("./entities/order-item.entity");
const postage_service_charge_entity_1 = require("./entities/postage-service-charge.entity");
const customer_entity_1 = require("../customers/entities/customer.entity");
const product_entity_1 = require("../products/entities/product.entity");
const user_entity_1 = require("../users/entities/user.entity");
let OrdersService = OrdersService_1 = class OrdersService {
    ordersRepository;
    orderItemsRepository;
    postageServiceChargesRepository;
    customersRepository;
    productsRepository;
    usersRepository;
    dataSource;
    logger = new common_1.Logger(OrdersService_1.name);
    constructor(ordersRepository, orderItemsRepository, postageServiceChargesRepository, customersRepository, productsRepository, usersRepository, dataSource) {
        this.ordersRepository = ordersRepository;
        this.orderItemsRepository = orderItemsRepository;
        this.postageServiceChargesRepository = postageServiceChargesRepository;
        this.customersRepository = customersRepository;
        this.productsRepository = productsRepository;
        this.usersRepository = usersRepository;
        this.dataSource = dataSource;
    }
    async generateOrderNumber() {
        const today = new Date();
        const dateStr = today.toISOString().split('T')[0].replace(/-/g, '');
        const lastOrder = await this.ordersRepository
            .createQueryBuilder('order')
            .where('order.orderNumber LIKE :pattern', { pattern: `ORD-${dateStr}-%` })
            .orderBy('order.createdAt', 'DESC')
            .getOne();
        let sequence = 1;
        if (lastOrder && lastOrder.orderNumber) {
            const parts = lastOrder.orderNumber.split('-');
            if (parts.length >= 3) {
                const lastSequence = parseInt(parts[2] || '0', 10);
                sequence = lastSequence + 1;
            }
        }
        const orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
        const exists = await this.ordersRepository.findOne({
            where: { orderNumber },
        });
        if (exists) {
            sequence++;
            return `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
        }
        return orderNumber;
    }
    async cleanupInvalidOrders() {
        try {
            await this.fixEmptyOrderNumbers();
            const allOrders = await this.ordersRepository.find();
            const allCustomerIds = (await this.customersRepository.find()).map(c => c.id);
            const invalidOrders = allOrders.filter(order => order.customerId !== null && order.customerId !== undefined && !allCustomerIds.includes(order.customerId));
            if (invalidOrders.length > 0) {
                console.log(`Found ${invalidOrders.length} orders with invalid customerIds. Removing...`);
                for (const order of invalidOrders) {
                    await this.orderItemsRepository.delete({ orderId: order.id });
                }
                await this.ordersRepository.remove(invalidOrders);
                console.log(`Removed ${invalidOrders.length} orders with invalid customerIds.`);
            }
        }
        catch (error) {
            console.warn('Failed to cleanup invalid orders:', error.message);
        }
    }
    async fixEmptyOrderNumbers() {
        try {
            const ordersWithoutNumber = await this.ordersRepository
                .createQueryBuilder('order')
                .where('order.orderNumber IS NULL OR order.orderNumber = :empty', { empty: '' })
                .getMany();
            if (ordersWithoutNumber.length > 0) {
                console.log(`Found ${ordersWithoutNumber.length} orders without order numbers. Fixing...`);
                for (const order of ordersWithoutNumber) {
                    const orderDate = order.createdAt || new Date();
                    const dateStr = orderDate.toISOString().split('T')[0].replace(/-/g, '');
                    const allOrders = await this.ordersRepository
                        .createQueryBuilder('o')
                        .where('o.orderNumber IS NOT NULL AND o.orderNumber != :empty', { empty: '' })
                        .andWhere('o.orderNumber LIKE :pattern', { pattern: `ORD-${dateStr}-%` })
                        .getMany();
                    let maxSequence = 0;
                    allOrders.forEach(o => {
                        if (o.orderNumber) {
                            const parts = o.orderNumber.split('-');
                            if (parts.length >= 3) {
                                const seq = parseInt(parts[2] || '0', 10);
                                if (seq > maxSequence)
                                    maxSequence = seq;
                            }
                        }
                    });
                    let sequence = maxSequence + 1;
                    let orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
                    let exists = await this.ordersRepository.findOne({
                        where: { orderNumber },
                    });
                    while (exists) {
                        sequence++;
                        orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
                        exists = await this.ordersRepository.findOne({
                            where: { orderNumber },
                        });
                    }
                    order.orderNumber = orderNumber;
                    await this.ordersRepository.save(order);
                }
                console.log(`Fixed ${ordersWithoutNumber.length} orders with missing order numbers.`);
            }
        }
        catch (error) {
            console.warn('Failed to fix empty order numbers:', error.message);
        }
    }
    async create(createOrderDto, user) {
        try {
            this.logger.debug(`create() - Creating order for customer ID: ${createOrderDto.customerId}, user ID: ${user.id}, items count: ${createOrderDto.orderItems?.length || 0}`);
            if (!createOrderDto.customerId) {
                this.logger.warn(`create() - BadRequest: Customer ID is required`);
                throw new common_1.BadRequestException('Customer ID is required');
            }
            const customer = await this.customersRepository.findOne({
                where: { id: createOrderDto.customerId },
            });
            if (!customer) {
                this.logger.warn(`create() - NotFoundException: Customer with ID ${createOrderDto.customerId} not found`);
                throw new common_1.NotFoundException(`Customer with ID ${createOrderDto.customerId} not found`);
            }
            this.logger.debug(`create() - Customer validated: ${customer.id}`);
            const productIds = createOrderDto.orderItems.map((item) => item.productId);
            this.logger.debug(`create() - Validating ${productIds.length} products: ${productIds.join(', ')}`);
            const products = await this.productsRepository.find({
                where: productIds.map((id) => ({ id })),
            });
            if (products.length !== productIds.length) {
                const foundIds = products.map((p) => p.id);
                const missingIds = productIds.filter((id) => !foundIds.includes(id));
                this.logger.warn(`create() - NotFoundException: Products with IDs ${missingIds.join(', ')} not found`);
                throw new common_1.NotFoundException(`Products with IDs ${missingIds.join(', ')} not found`);
            }
            this.logger.debug(`create() - All products validated`);
            const postageServiceChargeIds = createOrderDto.orderItems
                .map((item) => item.postageServiceChargeId)
                .filter((id) => id !== undefined && id !== null);
            if (postageServiceChargeIds.length > 0) {
                this.logger.debug(`create() - Validating ${postageServiceChargeIds.length} postage/service charges`);
                const uniqueIds = [...new Set(postageServiceChargeIds)];
                const charges = await this.postageServiceChargesRepository.find({
                    where: uniqueIds.map((id) => ({ id })),
                });
                if (charges.length !== uniqueIds.length) {
                    const foundIds = charges.map((c) => c.id);
                    const missingIds = uniqueIds.filter((id) => !foundIds.includes(id));
                    this.logger.warn(`create() - NotFoundException: Postage/Service charges with IDs ${missingIds.join(', ')} not found`);
                    throw new common_1.NotFoundException(`Postage/Service charges with IDs ${missingIds.join(', ')} not found`);
                }
                this.logger.debug(`create() - All postage/service charges validated`);
            }
            this.logger.debug(`create() - Generating order number`);
            const orderNumber = await this.generateOrderNumber();
            if (!orderNumber) {
                this.logger.error(`create() - BadRequestException: Failed to generate order number`);
                throw new common_1.BadRequestException('Failed to generate order number');
            }
            this.logger.debug(`create() - Generated order number: ${orderNumber}`);
            const orderDate = createOrderDto.orderDate
                ? new Date(createOrderDto.orderDate)
                : new Date();
            this.logger.debug(`create() - Creating order entity`);
            const order = this.ordersRepository.create({
                orderNumber,
                customerId: createOrderDto.customerId,
                createdById: user.id,
                orderDate,
            });
            const savedOrder = await this.ordersRepository.save(order);
            this.logger.debug(`create() - Order saved with ID: ${savedOrder.id}`);
            this.logger.debug(`create() - Creating ${createOrderDto.orderItems.length} order items`);
            const orderItems = createOrderDto.orderItems.map((item) => {
                const orderItem = new order_item_entity_1.OrderItem();
                orderItem.orderId = savedOrder.id;
                orderItem.productId = item.productId;
                orderItem.count = item.count;
                orderItem.amountPerTub = item.amountPerTub;
                orderItem.totalAmount = item.totalAmount;
                if (item.postageServiceChargeId !== undefined && item.postageServiceChargeId !== null) {
                    orderItem.postageServiceChargeId = item.postageServiceChargeId;
                }
                if (item.deductionNote !== undefined && item.deductionNote !== null) {
                    orderItem.deductionNote = item.deductionNote;
                }
                return orderItem;
            });
            await this.orderItemsRepository.save(orderItems);
            this.logger.debug(`create() - Order items saved`);
            const result = await this.findOne(savedOrder.id, user);
            this.logger.log(`create() - Successfully created order with ID: ${result.id}, orderNumber: ${result.orderNumber}`);
            return result;
        }
        catch (error) {
            const errorType = error.constructor.name;
            if (error instanceof common_1.NotFoundException || error instanceof common_1.BadRequestException) {
                this.logger.error(`create() - ${errorType}: ${error.message}`);
                throw error;
            }
            this.logger.error(`create() - Error creating order. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
            throw new common_1.BadRequestException(`Failed to create order: ${error.message || 'Unknown error occurred'}`);
        }
    }
    async findAll(listOrdersDto, user) {
        try {
            const { page = 1, limit = 10, customerId, orderNumber, createdById, startDate, endDate, } = listOrdersDto;
            const skip = (page - 1) * limit;
            this.logger.debug(`findAll() - Finding orders with filters: page=${page}, limit=${limit}, customerId=${customerId}, orderNumber=${orderNumber}, createdById=${createdById}, startDate=${startDate}, endDate=${endDate}, user ID: ${user.id}`);
            const queryBuilder = this.ordersRepository
                .createQueryBuilder('order')
                .leftJoinAndSelect('order.customer', 'customer')
                .leftJoinAndSelect('order.createdBy', 'createdBy')
                .leftJoinAndSelect('order.orderItems', 'orderItems')
                .leftJoinAndSelect('orderItems.product', 'product')
                .leftJoinAndSelect('orderItems.postageServiceCharge', 'postageServiceCharge');
            if (customerId) {
                queryBuilder.where('order.customerId = :customerId', { customerId });
                this.logger.debug(`findAll() - Applied customerId filter: ${customerId}`);
            }
            if (orderNumber) {
                if (customerId) {
                    queryBuilder.andWhere('order.orderNumber LIKE :orderNumber', {
                        orderNumber: `%${orderNumber}%`,
                    });
                }
                else {
                    queryBuilder.where('order.orderNumber LIKE :orderNumber', {
                        orderNumber: `%${orderNumber}%`,
                    });
                }
                this.logger.debug(`findAll() - Applied orderNumber filter: ${orderNumber}`);
            }
            if (createdById) {
                const whereClause = customerId || orderNumber ? 'andWhere' : 'where';
                queryBuilder[whereClause]('order.createdById = :createdById', {
                    createdById,
                });
                this.logger.debug(`findAll() - Applied createdById filter: ${createdById}`);
            }
            if (startDate && endDate) {
                const whereClause = customerId || orderNumber || createdById ? 'andWhere' : 'where';
                queryBuilder[whereClause]('order.createdAt BETWEEN :startDate AND :endDate', {
                    startDate,
                    endDate,
                });
                this.logger.debug(`findAll() - Applied date range filter: ${startDate} to ${endDate}`);
            }
            else if (startDate) {
                const whereClause = customerId || orderNumber || createdById ? 'andWhere' : 'where';
                queryBuilder[whereClause]('order.createdAt >= :startDate', { startDate });
                this.logger.debug(`findAll() - Applied startDate filter: ${startDate}`);
            }
            else if (endDate) {
                const whereClause = customerId || orderNumber || createdById ? 'andWhere' : 'where';
                queryBuilder[whereClause]('order.createdAt <= :endDate', { endDate });
                this.logger.debug(`findAll() - Applied endDate filter: ${endDate}`);
            }
            queryBuilder
                .andWhere('order.isActive = :isActive', { isActive: true })
                .skip(skip)
                .take(limit)
                .orderBy('order.createdAt', 'DESC');
            const [data, total] = await queryBuilder.getManyAndCount();
            this.logger.log(`findAll() - Successfully retrieved ${data.length} orders (total: ${total})`);
            return { data, total };
        }
        catch (error) {
            const errorType = error.constructor.name;
            this.logger.error(`findAll() - Error finding orders. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
            throw error;
        }
    }
    async findOne(id, user) {
        try {
            this.logger.debug(`findOne() - Searching for order with ID: ${id}, user ID: ${user.id}`);
            const order = await this.ordersRepository.findOne({
                where: { id },
                relations: [
                    'customer',
                    'createdBy',
                    'orderItems',
                    'orderItems.product',
                    'orderItems.postageServiceCharge',
                ],
            });
            if (!order) {
                this.logger.warn(`findOne() - Order with ID ${id} not found`);
                throw new common_1.NotFoundException(`Order with ID ${id} not found`);
            }
            this.logger.log(`findOne() - Successfully found order with ID: ${id}, orderNumber: ${order.orderNumber}`);
            return order;
        }
        catch (error) {
            const errorType = error.constructor.name;
            if (error instanceof common_1.NotFoundException) {
                throw error;
            }
            this.logger.error(`findOne() - Error finding order with ID ${id}. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
            throw error;
        }
    }
    async update(id, updateOrderDto, user) {
        try {
            this.logger.debug(`update() - Updating order with ID: ${id}, user ID: ${user.id}, data: ${JSON.stringify(updateOrderDto)}`);
            return await this.dataSource.transaction(async (manager) => {
                const orderRepository = manager.getRepository(order_entity_1.Order);
                const orderItemsRepository = manager.getRepository(order_item_entity_1.OrderItem);
                const postageServiceChargesRepository = manager.getRepository(postage_service_charge_entity_1.PostageServiceCharge);
                const customersRepository = manager.getRepository(customer_entity_1.Customer);
                const productsRepository = manager.getRepository(product_entity_1.Product);
                this.logger.debug(`update() - Fetching existing order with ID: ${id}`);
                const existingOrder = await orderRepository.findOne({
                    where: { id },
                    relations: ['customer', 'createdBy'],
                });
                if (!existingOrder) {
                    this.logger.warn(`update() - NotFoundException: Order with ID ${id} not found`);
                    throw new common_1.NotFoundException(`Order with ID ${id} not found`);
                }
                this.logger.debug(`update() - Found existing order: ${existingOrder.orderNumber}`);
                const existingOrderItems = await orderItemsRepository.find({
                    where: { orderId: id },
                });
                this.logger.debug(`update() - Found ${existingOrderItems.length} existing order items`);
                if (updateOrderDto.customerId !== undefined) {
                    this.logger.debug(`update() - Validating customer ID: ${updateOrderDto.customerId}`);
                    const customer = await customersRepository.findOne({
                        where: { id: updateOrderDto.customerId },
                    });
                    if (!customer) {
                        this.logger.warn(`update() - NotFoundException: Customer with ID ${updateOrderDto.customerId} not found`);
                        throw new common_1.NotFoundException(`Customer with ID ${updateOrderDto.customerId} not found`);
                    }
                    existingOrder.customerId = updateOrderDto.customerId;
                    this.logger.debug(`update() - Customer updated to: ${updateOrderDto.customerId}`);
                }
                if (updateOrderDto.orderDate !== undefined) {
                    existingOrder.orderDate = new Date(updateOrderDto.orderDate);
                }
                if (updateOrderDto.orderItems !== undefined) {
                    this.logger.debug(`update() - Updating ${updateOrderDto.orderItems.length} order items`);
                    const productIds = updateOrderDto.orderItems
                        .map((item) => item.productId)
                        .filter((id) => id !== undefined);
                    if (productIds.length > 0) {
                        this.logger.debug(`update() - Validating ${productIds.length} products: ${productIds.join(', ')}`);
                        const products = await productsRepository.find({
                            where: productIds.map((id) => ({ id })),
                        });
                        if (products.length !== productIds.length) {
                            const foundIds = products.map((p) => p.id);
                            const missingIds = productIds.filter((id) => !foundIds.includes(id));
                            this.logger.warn(`update() - NotFoundException: Products with IDs ${missingIds.join(', ')} not found`);
                            throw new common_1.NotFoundException(`Products with IDs ${missingIds.join(', ')} not found`);
                        }
                        this.logger.debug(`update() - All products validated`);
                    }
                    const postageServiceChargeIds = updateOrderDto.orderItems
                        .map((item) => item.postageServiceChargeId)
                        .filter((id) => id !== undefined && id !== null);
                    if (postageServiceChargeIds.length > 0) {
                        this.logger.debug(`update() - Validating ${postageServiceChargeIds.length} postage/service charges`);
                        const uniqueIds = [...new Set(postageServiceChargeIds)];
                        const charges = await postageServiceChargesRepository.find({
                            where: uniqueIds.map((id) => ({ id })),
                        });
                        if (charges.length !== uniqueIds.length) {
                            const foundIds = charges.map((c) => c.id);
                            const missingIds = uniqueIds.filter((id) => !foundIds.includes(id));
                            this.logger.warn(`update() - NotFoundException: Postage/Service charges with IDs ${missingIds.join(', ')} not found`);
                            throw new common_1.NotFoundException(`Postage/Service charges with IDs ${missingIds.join(', ')} not found`);
                        }
                        this.logger.debug(`update() - All postage/service charges validated`);
                    }
                    this.logger.debug(`update() - Checking for amount decreases and validating deduction notes`);
                    for (const item of updateOrderDto.orderItems) {
                        if (item.totalAmount !== undefined) {
                            const existingItem = existingOrderItems.find((oi) => oi.productId === item.productId);
                            if (existingItem && existingItem.totalAmount) {
                                const oldAmount = parseFloat(existingItem.totalAmount.toString());
                                const newAmount = parseFloat(item.totalAmount.toString());
                                if (newAmount < oldAmount) {
                                    if (!item.deductionNote || item.deductionNote.trim() === '') {
                                        this.logger.warn(`update() - BadRequestException: Deduction note required for product ID ${item.productId || existingItem.productId}, old amount: ${oldAmount}, new amount: ${newAmount}`);
                                        throw new common_1.BadRequestException(`Deduction note is required when decreasing total amount for product ID ${item.productId || existingItem.productId}. Old amount: ${oldAmount}, New amount: ${newAmount}`);
                                    }
                                    this.logger.debug(`update() - Deduction note provided for product ID ${item.productId || existingItem.productId}`);
                                }
                            }
                        }
                    }
                    this.logger.debug(`update() - Removing existing order items`);
                    await orderItemsRepository.delete({ orderId: id });
                    this.logger.debug(`update() - Creating new order items`);
                    const orderItems = updateOrderDto.orderItems.map((item) => {
                        if (!item.productId) {
                            this.logger.warn(`update() - BadRequestException: Product ID is required for order items`);
                            throw new common_1.BadRequestException('Product ID is required for order items');
                        }
                        const orderItem = new order_item_entity_1.OrderItem();
                        orderItem.orderId = id;
                        orderItem.productId = item.productId;
                        orderItem.count = item.count !== undefined ? item.count : 1;
                        if (item.amountPerTub !== undefined) {
                            orderItem.amountPerTub = item.amountPerTub;
                        }
                        else {
                            this.logger.warn(`update() - BadRequestException: amountPerTub is required for order items`);
                            throw new common_1.BadRequestException('amountPerTub is required for order items');
                        }
                        if (item.totalAmount !== undefined) {
                            orderItem.totalAmount = item.totalAmount;
                        }
                        else {
                            this.logger.warn(`update() - BadRequestException: totalAmount is required for order items`);
                            throw new common_1.BadRequestException('totalAmount is required for order items');
                        }
                        if (item.postageServiceChargeId !== undefined && item.postageServiceChargeId !== null) {
                            orderItem.postageServiceChargeId = item.postageServiceChargeId;
                        }
                        if (item.deductionNote !== undefined && item.deductionNote !== null) {
                            orderItem.deductionNote = item.deductionNote;
                        }
                        return orderItem;
                    });
                    await orderItemsRepository.insert(orderItems.map(item => {
                        const insertData = {
                            orderId: item.orderId,
                            productId: item.productId,
                            count: item.count,
                            amountPerTub: item.amountPerTub,
                            totalAmount: item.totalAmount,
                        };
                        if (item.postageServiceChargeId !== undefined && item.postageServiceChargeId !== null) {
                            insertData.postageServiceChargeId = item.postageServiceChargeId;
                        }
                        if (item.deductionNote !== undefined && item.deductionNote !== null) {
                            insertData.deductionNote = item.deductionNote;
                        }
                        return insertData;
                    }));
                }
                const orderUpdateData = {};
                if (updateOrderDto.isActive !== undefined) {
                    orderUpdateData.isActive = updateOrderDto.isActive;
                }
                if (updateOrderDto.customerId !== undefined) {
                    orderUpdateData.customerId = updateOrderDto.customerId;
                }
                if (updateOrderDto.orderDate !== undefined) {
                    orderUpdateData.orderDate = new Date(updateOrderDto.orderDate);
                }
                await orderRepository.update(id, orderUpdateData);
                const result = await this.findOne(id, user);
                this.logger.log(`update() - Successfully updated order with ID: ${id}, orderNumber: ${result.orderNumber}`);
                return result;
            });
        }
        catch (error) {
            const errorType = error.constructor.name;
            if (error instanceof common_1.NotFoundException || error instanceof common_1.BadRequestException) {
                this.logger.error(`update() - ${errorType}: ${error.message}`);
                throw error;
            }
            this.logger.error(`update() - Error updating order with ID ${id}. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
            throw new common_1.BadRequestException(`Failed to update order: ${error.message || 'Unknown error occurred'}`);
        }
    }
    async remove(id, user) {
        try {
            this.logger.debug(`remove() - Deleting order with ID: ${id}, user ID: ${user.id}`);
            const order = await this.findOne(id, user);
            order.isActive = false;
            await this.ordersRepository.save(order);
            this.logger.log(`remove() - Successfully deleted order with ID: ${id}, orderNumber: ${order.orderNumber}`);
        }
        catch (error) {
            const errorType = error.constructor.name;
            if (error instanceof common_1.NotFoundException) {
                throw error;
            }
            this.logger.error(`remove() - Error deleting order with ID ${id}. Error Type: ${errorType}, Message: ${error.message}`, error.stack);
            throw error;
        }
    }
};
exports.OrdersService = OrdersService;
exports.OrdersService = OrdersService = OrdersService_1 = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(order_entity_1.Order)),
    __param(1, (0, typeorm_1.InjectRepository)(order_item_entity_1.OrderItem)),
    __param(2, (0, typeorm_1.InjectRepository)(postage_service_charge_entity_1.PostageServiceCharge)),
    __param(3, (0, typeorm_1.InjectRepository)(customer_entity_1.Customer)),
    __param(4, (0, typeorm_1.InjectRepository)(product_entity_1.Product)),
    __param(5, (0, typeorm_1.InjectRepository)(user_entity_1.User)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.DataSource])
], OrdersService);
//# sourceMappingURL=orders.service.js.map