import {
  Injectable,
  NotFoundException,
  InternalServerErrorException,
  ConflictException,
  BadRequestException,
  Logger,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, DataSource } from 'typeorm';
import * as XLSX from 'xlsx';
import { User } from './entities/user.entity';
import { CreateUserDto } from './dto/create-user.dto';
import { UpdateUserDto } from './dto/update-user.dto';
import { EmailService } from '../common/services/email.service';
import { firstValueFrom } from 'rxjs';
import { UserActivity } from '../user-activity/entities/user-activity.entity';
import { PermissionsService } from '../permissions/permissions.service';
import { Login } from '../login/entities/login.entity';
import { PermissionResponse } from '../permissions/permissions.model';

interface ExcelRecord {
  email?: string;
  firstName?: string;
  lastName?: string;
  role?: string;
  [key: string]: any;
}

@Injectable()
export class UsersService {
  private readonly logger = new Logger(UsersService.name);

  constructor(
    @InjectRepository(User)
    private usersRepository: Repository<User>,

    @InjectRepository(UserActivity)
    private activityRepository: Repository<UserActivity>,

    @InjectRepository(Login)
    private loginRepository: Repository<Login>,

    private emailService: EmailService,
    private permissionsService: PermissionsService,
    private dataSource: DataSource,
  ) {}

  private sanitizeUser(user: User): Partial<User> {
    const { emailVerified, lastLoginAt, createdAt, updatedAt, ...rest } = user;
    return rest;
  }

  async logActivity(data: Partial<UserActivity>) {
    const activity = this.activityRepository.create(data);
    return await this.activityRepository.save(activity);
  }

  private async recordActivity(params: {
    user: User;
    action: string;
    type: string;
    method: string;
    changedObject: string;
    ipAddress: string;
  }) {
    const { user, action, type, method, changedObject, ipAddress } = params;

    await this.logActivity({
      username: user.email,
      action,
      type,
      method,
      changedObject,
      ipAddress,
      user,
    });
  }

  private getRoleName(roleId: string): string {
    const roleMap: { [key: string]: string } = {
      '1': 'Admin',
      '2': 'Retention',
      '3': 'Reactivation',
      '4': 'Owner',
      '5': 'Escalator',
      '6': 'Agent',
      '7': 'Team Lead'
    };
    return roleMap[roleId] || 'Agent';
  }

  private getRoleIdFromName(roleName: string): string | null {
    const roleMap: { [key: string]: string } = {
      'Admin': '1',
      'Retention': '2',
      'Reactivation': '3',
      'Owner': '4',
      'Escalator': '5',
      'Agent': '6',
      'Team Lead': '7'
    };
    return roleMap[roleName] || null;
  }

  async create(createUserDto: CreateUserDto, creatorId: number): Promise<any> {
    return await this.dataSource.transaction(async manager => {
      const existingUser = await manager.findOne(User, { where: { email: createUserDto.user.email } });
      if (existingUser) throw new ConflictException('Email already exists');

      const roleName = this.getRoleName(createUserDto.role_id);
      const user = manager.create(User, {
        email: createUserDto.user.email,
        firstName: createUserDto.user.name.split(' ')[0] || createUserDto.user.name,
        lastName: createUserDto.user.name.split(' ').slice(1).join(' ') || '',
        role: roleName,
        isActive: true,
        status: 1,
        created_by: creatorId,
        permissions: [],
      });

      if (createUserDto.reports_to_id) {
        const reportsTo = await manager.findOne(User, { where: { id: parseInt(createUserDto.reports_to_id) } });
        if (reportsTo) {
          user.reportsTo = reportsTo;
          this.emailService.sendReportsToNotification(
            reportsTo.email,
            `${reportsTo.firstName} ${reportsTo.lastName}`,
            `${user.firstName} ${user.lastName}`,
            user.email
          ).catch(e => this.logger.error('Failed to send reports_to notification email:', e));
        }
      }

      const savedUser = await manager.save(User, user);

      const permissionsResponse = await this.permissionsService.findAll(createUserDto.role_id);
      // Combine all permissions for the role
      const allRolePermissions = [
        ...permissionsResponse.data,
        ...permissionsResponse.customer_permissions,
        ...permissionsResponse.user_permissions
      ];
      let grantedPermissions = allRolePermissions.filter(permission => 
        createUserDto.permissions.includes(permission.id.toString())
      );
      if (!Array.isArray(grantedPermissions)) grantedPermissions = [];
      savedUser.permissions = grantedPermissions;
      await manager.save(User, savedUser);

      this.emailService.sendWelcomeEmail(savedUser.email).catch(e => this.logger.error('Failed to send welcome email:', e));

      this.recordActivity({
        user: savedUser,
        action: 'Create',
        type: 'User',
        method: 'POST',
        changedObject: `Created user ${savedUser.email}`,
        ipAddress: '127.0.0.1',
      }).catch(e => this.logger.error('Failed to record activity:', e));

      const userWithReportsTo = await manager.findOne(User, { where: { id: savedUser.id }, relations: ['reportsTo'] });
      if (!userWithReportsTo) throw new InternalServerErrorException('Failed to fetch created user');

      return {
        data: {
          id: savedUser.id,
          user: {
            email: savedUser.email,
            name: `${savedUser.firstName} ${savedUser.lastName}`.trim(),
            last_login: savedUser.lastLoginAt?.toISOString() || null,
          },
          granted_permissions: grantedPermissions,
          role: {
            id: parseInt(createUserDto.role_id),
            name: roleName,
            permissions: allRolePermissions,
          },
          reports_to: userWithReportsTo.reportsTo ? {
            id: userWithReportsTo.reportsTo.id,
            user: {
              email: userWithReportsTo.reportsTo.email,
              name: `${userWithReportsTo.reportsTo.firstName} ${userWithReportsTo.reportsTo.lastName}`,
              last_login: userWithReportsTo.reportsTo.lastLoginAt?.toISOString() || null,
            },
            granted_permissions: userWithReportsTo.reportsTo.permissions || [],
          } : null,
          created_at: savedUser.createdAt,
          updated_at: savedUser.updatedAt,
          is_active: savedUser.isActive,
        },
        success: true,
        message: 'ok',
        status: 201,
      };
    });
  }

  async findAll(
    limit: number = 10,
    offset: number = 0,
    search: string = '',
    isActive?: boolean,
    requestingUser?: User,
  ): Promise<{
    next: string | null;
    previous: string | null;
    count: number;
    limit: number;
    offset: number;
    status: number;
    success: boolean;
    message: string;
    data: Array<{
      id: number;
      user: {
        email: string;
        name: string;
        last_login: string | null;
      };
      role: {
        id: number;
        name: string;
        permissions: Array<{
          id: number;
          name: string;
          codename: string;
          content_type: number;
        }>;
      };
      reports_to: {
        id: number;
        user: {
          email: string;
          name: string;
          last_login: string | null;
        };
        granted_permissions: Array<{
          id: number;
          name: string;
          codename: string;
          content_type: number;
        }>;
      } | null;
      granted_permissions: Array<{
        id: number;
        name: string;
        codename: string;
        content_type: number;
      }>;
      is_active: boolean;
    }>;
  }> {
    try {
      // Build query with search and pagination
      const queryBuilder = this.usersRepository.createQueryBuilder('user')
        .leftJoinAndSelect('user.reportsTo', 'reportsTo');
      
      // Add search conditions
      if (search) {
        queryBuilder.where(
          '(LOWER(user.email) LIKE LOWER(:search) OR LOWER(user.firstName) LIKE LOWER(:search) OR LOWER(user.lastName) LIKE LOWER(:search))',
          { search: `%${search}%` }
        );
      }

      // Add isActive filter if provided
      if (typeof isActive === 'boolean') {
        queryBuilder.andWhere('user.isActive = :isActive', { isActive });
      }

     
       // Add role-based filtering
      if (requestingUser) {
        if (requestingUser.role !== 'Admin' && requestingUser.role !== 'Owner') {
          // Non-admin/owner roles can see:
          // 1. Users they created
          // 2. Users who report to them
          queryBuilder.andWhere(
            '(user.created_by = :creatorId OR user.reports_to_id = :userId)',
            { 
              creatorId: requestingUser.id,
              userId: requestingUser.id
            }
          );
          // Non-admin/owner roles cannot see Admin or Owner users
          queryBuilder.andWhere('user.role NOT IN (:...restrictedRoles)', { restrictedRoles: ['Admin', 'Owner'] });
        }
        // Admin and Owner can see all users (no additional conditions needed)
      }

      // Get total count
      const totalCount = await queryBuilder.getCount();

      // Apply pagination
      queryBuilder
        .skip(offset)
        .take(limit);

      const users = await queryBuilder.getMany();
      this.logger.debug(`Found ${users.length} users`);

      // Get latest login for each user
      const userLogins = await Promise.all(
        users.map(async (user) => {
          const latestLogin = await this.loginRepository.findOne({
            where: { email: user.email, status: 'success' },
            order: { verifiedAt: 'DESC' },
          });
          return { userId: user.id, login: latestLogin };
        })
      );

      // Format the response
      const formattedData = users.map(user => {
        const roleId = this.getRoleIdFromName(user.role);
        const userLogin = userLogins.find(ul => ul.userId === user.id)?.login;
        
        return {
          id: user.id,
          user: {
            email: user.email,
            name: `${user.firstName} ${user.lastName}`,
            last_login: userLogin?.verifiedAt?.toISOString() || user.lastLoginAt?.toISOString() || null,
          },
          role: {
            id: roleId ? parseInt(roleId) : 0,
            name: user.role,
            permissions: [], // We'll fetch permissions separately if needed
          },
          reports_to: user.reportsTo ? {
            id: user.reportsTo.id,
            user: {
              email: user.reportsTo.email,
              name: `${user.reportsTo.firstName} ${user.reportsTo.lastName}`,
              last_login: user.reportsTo.lastLoginAt?.toISOString() || null,
            },
            granted_permissions: user.reportsTo.permissions || [],
          } : null,
          granted_permissions: user.permissions || [],
          is_active: user.isActive,
        };
      });

      // Calculate next and previous URLs
      const baseUrl = process.env.API_URL || 'http://localhost:3000';
      const next = offset + limit < totalCount 
        ? `${baseUrl}/api/users/employees/?limit=${limit}&offset=${offset + limit}&search=${search}&isActive=${isActive}`
        : null;
      const previous = offset > 0
        ? `${baseUrl}/api/users/employees/?limit=${limit}&offset=${Math.max(0, offset - limit)}&search=${search}&isActive=${isActive}`
        : null;

      return {
        next,
        previous,
        count: totalCount,
        limit,
        offset,
        status: 200,
        success: true,
        message: 'List Successfully Retrieved!',
        data: formattedData,
      };
    } catch (error) {
      this.logger.error('Error finding users:', error);
      throw new InternalServerErrorException('Failed to fetch users');
    }
  }

  async findOne(id: number, requestingUser?: User): Promise<any> {
    if (!id || isNaN(id)) throw new BadRequestException('Invalid user ID');

    try {
      const user = await this.usersRepository.findOne({ 
        where: { id },
        relations: ['reportsTo']
      });
      
      if (!user) throw new NotFoundException(`User with ID ${id} not found`);

      // Check permissions
      if (requestingUser) {
        if (requestingUser.role !== 'Admin' && requestingUser.role !== 'Owner') {
          // Non-admin/owner roles can see:
          // 1. Users they created
          // 2. Users who report to them
          if (user.created_by !== requestingUser.id && (!user.reportsTo || user.reportsTo.id !== requestingUser.id)) {
            throw new NotFoundException(`User with ID ${id} not found`);
          }
          // Non-admin/owner roles cannot see Admin or Owner users
          if (user.role === 'Admin' || user.role === 'Owner') {
            throw new NotFoundException(`User with ID ${id} not found`);
          }
        }
        // Admin and Owner can see all users (no additional conditions needed)
      }

      // Get role ID from role name
      const roleId = this.getRoleIdFromName(user.role) || '1'; // Default to '1' if role not found
      
      // Fetch permissions for the role
      const permissionsResponse = await this.permissionsService.findAll(roleId);
      const rolePermissions = permissionsResponse.data;

      // Parse the stored permissions if they exist
      let grantedPermissions: PermissionResponse[] = [];
      if (user.permissions) {
        try {
          // If permissions is a string, parse it
          if (typeof user.permissions === 'string') {
            grantedPermissions = JSON.parse(user.permissions);
          } else {
            grantedPermissions = user.permissions;
          }
        } catch (error) {
          this.logger.error(`Error parsing permissions for user ${id}:`, error);
          grantedPermissions = [];
        }
      }

      // Transform the user data into the required format
      return {
        id: user.id,
        user: {
          email: user.email,
          name: `${user.firstName} ${user.lastName}`,
          last_login: user.lastLoginAt?.toISOString() || null
        },
        granted_permissions: grantedPermissions,
        role: {
          id: parseInt(roleId),
          name: user.role,
          permissions: rolePermissions
        },
        reports_to: user.reportsTo ? {
          id: user.reportsTo.id,
          user: {
            email: user.reportsTo.email,
            name: `${user.reportsTo.firstName} ${user.reportsTo.lastName}`,
            last_login: user.reportsTo.lastLoginAt?.toISOString() || null
          },
          granted_permissions: user.reportsTo.permissions || []
        } : null,
        created_at: user.createdAt,
        updated_at: user.updatedAt,
        is_active: user.isActive
      };
    } catch (error) {
      this.logger.error(`Error finding user with ID ${id}:`, error);
      throw error instanceof NotFoundException ? error : new InternalServerErrorException();
    }
  }

  async findByEmail(email: string): Promise<User | null> {
    if (!email) throw new BadRequestException('Email is required');
    try {
      return await this.usersRepository.findOne({ where: { email } });
    } catch (error) {
      this.logger.error('Error finding user by email:', error);
      throw new InternalServerErrorException('Failed to fetch user by email');
    }
  }

  async update(id: number, updateUserDto: UpdateUserDto): Promise<any> {
    return await this.dataSource.transaction(async manager => {
      const existingUser = await manager.findOne(User, { where: { id }, relations: ['reportsTo'] });
      if (!existingUser) {
        throw new NotFoundException(`User with ID ${id} not found`);
      }

      if (!Array.isArray(existingUser.permissions)) {
        existingUser.permissions = [];
      }

      if (updateUserDto.user) {
        existingUser.email = updateUserDto.user.email;
        const nameParts = updateUserDto.user.name.split(' ');
        existingUser.firstName = nameParts[0] || updateUserDto.user.name;
        existingUser.lastName = nameParts.slice(1).join(' ') || '';
      }

      if (updateUserDto.role_id) {
        existingUser.role = this.getRoleName(updateUserDto.role_id);
      }

      if (updateUserDto.reports_to_id !== undefined) {
        if (updateUserDto.reports_to_id) {
          const reportsTo = await manager.findOne(User, { where: { id: parseInt(updateUserDto.reports_to_id) } });
          if (reportsTo) {
            existingUser.reportsTo = reportsTo;
            this.emailService.sendReportsToNotification(
              reportsTo.email,
              `${reportsTo.firstName} ${reportsTo.lastName}`,
              `${existingUser.firstName} ${existingUser.lastName}`,
              existingUser.email
            ).catch(e => this.logger.error('Failed to send reports_to notification email:', e));
          }
        }
      }

      if (updateUserDto.permissions && updateUserDto.permissions.length > 0) {
        const allPermissions = await this.permissionsService.getAllPermissions();
        const grantedPermissions = allPermissions.filter(permission => 
          updateUserDto.permissions?.includes(permission.id.toString())
        );
        existingUser.permissions = Array.isArray(grantedPermissions) ? grantedPermissions : [];
      }

      const updatedUser = await manager.save(User, existingUser);

      this.recordActivity({
        user: updatedUser,
        action: 'Update',
        type: 'User',
        method: 'PATCH',
        changedObject: `Updated user ${updatedUser.email}`,
        ipAddress: '127.0.0.1',
      }).catch(e => this.logger.error('Failed to record activity:', e));

      const roleId = this.getRoleIdFromName(updatedUser.role) || '1';
      const permissionsResponse = await this.permissionsService.findAll(roleId);
      const rolePermissions = permissionsResponse.data;

      return {
        id: updatedUser.id,
        user: {
          email: updatedUser.email,
          name: `${updatedUser.firstName} ${updatedUser.lastName}`.trim(),
          last_login: updatedUser.lastLoginAt?.toISOString() || null,
        },
        granted_permissions: updatedUser.permissions || [],
        role: {
          id: parseInt(roleId),
          name: updatedUser.role,
          permissions: rolePermissions,
        },
        reports_to: updatedUser.reportsTo ? {
          id: updatedUser.reportsTo.id,
          user: {
            email: updatedUser.reportsTo.email,
            name: `${updatedUser.reportsTo.firstName} ${updatedUser.reportsTo.lastName}`.trim(),
            last_login: updatedUser.reportsTo.lastLoginAt?.toISOString() || null,
          },
          granted_permissions: updatedUser.reportsTo.permissions || [],
        } : null,
        created_at: updatedUser.createdAt,
        updated_at: updatedUser.updatedAt,
        is_active: updatedUser.isActive,
      };
    });
  }

  async remove(id: number): Promise<{ status: string; message: string }> {
    try {
      const user = await this.usersRepository.findOne({ where: { id } });
      if (!user) {
        throw new NotFoundException(`User with ID ${id} not found`);
      }

      // Instead of deleting, deactivate the user
      user.isActive = false;
      await this.usersRepository.save(user);

      await this.recordActivity({
        user,
        action: 'Delete',
        type: 'User',
        method: 'DELETE',
        changedObject: `Deleted user ${user.email}`,
        ipAddress: '127.0.0.1',
      });

      return { status: 'success', message: `User has been successfully deactivated` };
    } catch (error) {
      this.logger.error(`Error removing user with ID ${id}:`, error);
      if (error instanceof NotFoundException) {
        throw error;
      }
      throw new InternalServerErrorException('Failed to remove user');
    }
  }

  async deactivate(id: number): Promise<{ user: Partial<User>; message: string }> {
    try {
      const user = await this.findOne(id);
      user.isActive = false;
      const updatedUser = await this.usersRepository.save(user);

      await this.recordActivity({
        user: updatedUser,
        action: 'Deactivate',
        type: 'User',
        method: 'PATCH',
        changedObject: `Deactivated user ${updatedUser.email}`,
        ipAddress: '127.0.0.1',
      });

      return { user: this.sanitizeUser(updatedUser), message: 'User is deactivated' };
    } catch (error) {
      this.logger.error(`Error deactivating user with ID ${id}:`, error);
      throw new InternalServerErrorException('Failed to deactivate user');
    }
  }

  async reactivate(id: number): Promise<{ user: Partial<User>; message: string }> {
    try {
      const user = await this.usersRepository.findOne({ where: { id } });
      if (!user) {
        throw new NotFoundException(`User with ID ${id} not found`);
      }
      
      user.isActive = true;
      const updatedUser = await this.usersRepository.save(user);

      await this.recordActivity({
        user: updatedUser,
        action: 'Reactivate',
        type: 'User',
        method: 'PATCH',
        changedObject: `Reactivated user ${updatedUser.email}`,
        ipAddress: '127.0.0.1',
      });

      return { user: this.sanitizeUser(updatedUser), message: 'User is reactivated' };
    } catch (error) {
      this.logger.error(`Error reactivating user with ID ${id}:`, error);
      throw new InternalServerErrorException('Failed to reactivate user');
    }
  }

  async findDeactivatedUsers(
    requestingUser?: User,
    ipAddress: string = '127.0.0.1',
  ): Promise<User[]> {
    try {
      const deactivatedUsers = await this.usersRepository.find({ where: { isActive: false } });
      this.logger.log(`Found ${deactivatedUsers.length} deactivated users`);

      if (requestingUser) {
        await this.recordActivity({
          user: requestingUser,
          action: 'View Deactivated List',
          type: 'User List',
          method: 'GET',
          changedObject: `Admin ${requestingUser.email} retrieved ${deactivatedUsers.length} deactivated users`,
          ipAddress,
        });
      } else {
        this.logger.warn('No requesting user provided for logging deactivated users list access');
      }

      return deactivatedUsers;
    } catch (error) {
      this.logger.error('Error finding deactivated users:', error);
      throw new InternalServerErrorException('Failed to fetch deactivated users');
    }
  }

  async bulkCreate(file: Express.Multer.File): Promise<{
    message: string;
    details: { success: number; failed: number; errors: string[] };
  }> {
    const results = { success: 0, failed: 0, errors: [] as string[] };

    try {
      const workbook = XLSX.read(file.buffer, { type: 'buffer' });
      const worksheet = workbook.Sheets[workbook.SheetNames[0]];
      const records = XLSX.utils.sheet_to_json<ExcelRecord>(worksheet);

      if (records.length === 0) throw new BadRequestException('The Excel file is empty');

      const requiredColumns = ['email', 'firstName', 'lastName', 'role'];
      const missingColumns = requiredColumns.filter(col => !(col in records[0]));
      if (missingColumns.length > 0) throw new BadRequestException(`Missing columns: ${missingColumns.join(', ')}`);

      for (const record of records) {
        try {
          // Find the role ID from the role name in the excel record
          const roleId = this.getRoleIdFromName(record.role?.trim() || '');
          
          if (!roleId) {
             throw new BadRequestException(`Invalid role specified: ${record.role}`);
          }

          const dto: CreateUserDto = {
            permissions: [], // Default empty permissions array for bulk create
            user: {
              email: record.email?.trim() || '',
              name: `${record.firstName?.trim() || ''} ${record.lastName?.trim() || ''}`.trim(),
            },
            role_id: roleId,
          };

          if (!dto.user.email || !dto.user.name || !dto.role_id) {
            throw new BadRequestException('All fields (email, firstName, lastName, role) are required');
          }

          await this.create(dto, 1);
          results.success++;
        } catch (error) {
          results.failed++;
          results.errors.push(`Row ${results.success + results.failed}: ${error.message}`);
        }
      }

      const message =
        results.failed === 0
          ? `Successfully created ${results.success} users.`
          : `Partially successful: ${results.success} created, ${results.failed} failed.`;

      return { message, details: results };
    } catch (error) {
      this.logger.error('Error processing Excel file:', error);
      throw new BadRequestException(error.message || 'Invalid Excel file');
    }
  }
}