import { Test, TestingModule } from '@nestjs/testing';
import { getRepositoryToken } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { OrdersService } from './orders.service';
import { Order } from './entities/order.entity';
import { OrderItem } from './entities/order-item.entity';
import { Customer } from '../customers/entities/customer.entity';
import { Product } from '../products/entities/product.entity';
import { User } from '../users/entities/user.entity';
import { NotFoundException, BadRequestException } from '@nestjs/common';
import { CreateOrderDto } from './dto/create-order.dto';

describe('OrdersService', () => {
  let service: OrdersService;
  let ordersRepository: Repository<Order>;
  let orderItemsRepository: Repository<OrderItem>;
  let customersRepository: Repository<Customer>;
  let productsRepository: Repository<Product>;
  let usersRepository: Repository<User>;

  const mockOrderRepository = {
    create: jest.fn(),
    save: jest.fn(),
    find: jest.fn(),
    findOne: jest.fn(),
    remove: jest.fn(),
    createQueryBuilder: jest.fn(),
  };

  const mockOrderItemRepository = {
    create: jest.fn(),
    save: jest.fn(),
    delete: jest.fn(),
  };

  const mockCustomerRepository = {
    findOne: jest.fn(),
    find: jest.fn(),
  };

  const mockProductRepository = {
    find: jest.fn(),
  };

  const mockUserRepository = {
    findOne: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        OrdersService,
        {
          provide: getRepositoryToken(Order),
          useValue: mockOrderRepository,
        },
        {
          provide: getRepositoryToken(OrderItem),
          useValue: mockOrderItemRepository,
        },
        {
          provide: getRepositoryToken(Customer),
          useValue: mockCustomerRepository,
        },
        {
          provide: getRepositoryToken(Product),
          useValue: mockProductRepository,
        },
        {
          provide: getRepositoryToken(User),
          useValue: mockUserRepository,
        },
      ],
    }).compile();

    service = module.get<OrdersService>(OrdersService);
    ordersRepository = module.get<Repository<Order>>(getRepositoryToken(Order));
    orderItemsRepository = module.get<Repository<OrderItem>>(
      getRepositoryToken(OrderItem),
    );
    customersRepository = module.get<Repository<Customer>>(
      getRepositoryToken(Customer),
    );
    productsRepository = module.get<Repository<Product>>(
      getRepositoryToken(Product),
    );
    usersRepository = module.get<Repository<User>>(getRepositoryToken(User));

    // Reset mocks
    jest.clearAllMocks();
  });

  it('should be defined', () => {
    expect(service).toBeDefined();
  });

  describe('cleanupInvalidOrders', () => {
    it('should remove orders with invalid customerIds', async () => {
      const validCustomer = { id: 1, firstName: 'John', lastName: 'Doe' };
      const invalidOrder = {
        id: 1,
        customerId: 999, // Invalid customer ID
        orderNumber: 'ORD-20240121-0001',
        createdAt: new Date(),
      };
      const validOrder = {
        id: 2,
        customerId: 1, // Valid customer ID
        orderNumber: 'ORD-20240121-0002',
        createdAt: new Date(),
      };

      mockCustomerRepository.find.mockResolvedValue([validCustomer]);
      mockOrderRepository.find.mockResolvedValue([invalidOrder, validOrder]);
      mockOrderItemRepository.delete.mockResolvedValue({ affected: 0 });
      mockOrderRepository.remove.mockResolvedValue([invalidOrder]);

      await service.cleanupInvalidOrders();

      expect(mockOrderRepository.remove).toHaveBeenCalledWith([invalidOrder]);
      expect(mockOrderItemRepository.delete).toHaveBeenCalledWith({
        orderId: 1,
      });
    });

    it('should fix empty order numbers', async () => {
      const orderWithoutNumber = {
        id: 1,
        customerId: 1,
        orderNumber: null,
        createdAt: new Date('2024-01-21'),
      };

      const queryBuilder = {
        where: jest.fn().mockReturnThis(),
        getMany: jest.fn().mockResolvedValue([orderWithoutNumber]),
        orderBy: jest.fn().mockReturnThis(),
      };

      mockOrderRepository.createQueryBuilder.mockReturnValue(queryBuilder);
      mockOrderRepository.findOne.mockResolvedValue(null);
      mockOrderRepository.save.mockResolvedValue({
        ...orderWithoutNumber,
        orderNumber: 'ORD-20240121-0001',
      });

      await service.cleanupInvalidOrders();

      expect(mockOrderRepository.save).toHaveBeenCalled();
    });
  });

  describe('create', () => {
    const mockUser: User = {
      id: 1,
      email: 'test@example.com',
      firstName: 'Test',
      lastName: 'User',
    } as User;

    const createOrderDto: CreateOrderDto = {
      customerId: 1,
      orderItems: [
        { productId: 1, count: 2 },
        { productId: 2, count: 1 },
      ],
    };

    it('should create an order successfully', async () => {
      const mockCustomer = { id: 1, firstName: 'John', lastName: 'Doe' };
      const mockProducts = [
        { id: 1, name: 'Product 1' },
        { id: 2, name: 'Product 2' },
      ];
      const mockOrder = {
        id: 1,
        orderNumber: 'ORD-20240121-0001',
        customerId: 1,
        createdById: 1,
      };
      const mockSavedOrder = {
        ...mockOrder,
        customer: mockCustomer,
        createdBy: mockUser,
        orderItems: [],
      };

      mockCustomerRepository.findOne.mockResolvedValue(mockCustomer);
      mockProductRepository.find.mockResolvedValue(mockProducts);
      mockOrderRepository.create.mockReturnValue(mockOrder);
      mockOrderRepository.save.mockResolvedValueOnce(mockOrder);
      mockOrderItemRepository.create.mockReturnValue({});
      mockOrderItemRepository.save.mockResolvedValue([]);
      mockOrderRepository.findOne.mockResolvedValueOnce(mockSavedOrder);

      const result = await service.create(createOrderDto, mockUser);

      expect(result).toBeDefined();
      expect(result.orderNumber).toBeDefined();
      expect(mockOrderRepository.save).toHaveBeenCalled();
    });

    it('should throw NotFoundException if customer does not exist', async () => {
      mockCustomerRepository.findOne.mockResolvedValue(null);

      await expect(service.create(createOrderDto, mockUser)).rejects.toThrow(
        NotFoundException,
      );
    });

    it('should throw NotFoundException if product does not exist', async () => {
      const mockCustomer = { id: 1, firstName: 'John', lastName: 'Doe' };
      mockCustomerRepository.findOne.mockResolvedValue(mockCustomer);
      mockProductRepository.find.mockResolvedValue([{ id: 1 }]);

      await expect(service.create(createOrderDto, mockUser)).rejects.toThrow(
        NotFoundException,
      );
    });

    it('should throw BadRequestException if customerId is not provided', async () => {
      const invalidDto = { ...createOrderDto, customerId: null };

      await expect(service.create(invalidDto as any, mockUser)).rejects.toThrow(
        BadRequestException,
      );
    });
  });

  describe('findOne', () => {
    it('should return an order if found', async () => {
      const mockOrder = {
        id: 1,
        orderNumber: 'ORD-20240121-0001',
        customerId: 1,
        customer: { id: 1, firstName: 'John' },
        createdBy: { id: 1, firstName: 'User' },
        orderItems: [],
      };
      const mockUser = { id: 1 } as User;

      mockOrderRepository.findOne.mockResolvedValue(mockOrder);

      const result = await service.findOne(1, mockUser);

      expect(result).toEqual(mockOrder);
      expect(mockOrderRepository.findOne).toHaveBeenCalledWith({
        where: { id: 1 },
        relations: [
          'customer',
          'createdBy',
          'orderItems',
          'orderItems.product',
        ],
      });
    });

    it('should throw NotFoundException if order not found', async () => {
      mockOrderRepository.findOne.mockResolvedValue(null);
      const mockUser = { id: 1 } as User;

      await expect(service.findOne(999, mockUser)).rejects.toThrow(
        NotFoundException,
      );
    });
  });
});

