import { Controller, Post, Get, Body, Query, HttpStatus, UseGuards, ParseIntPipe, DefaultValuePipe } from '@nestjs/common';
import { CustomerNotesService } from './customer-notes.service';
import { CreateCustomerNoteDto } from './dto/create-customer-note.dto';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { PermissionsGuard } from '../auth/guards/permissions.guard';
import { User } from '../auth/decorators/user.decorator';

@Controller([
  'api/customer/notes',
  'api/customer/notes/',
  'api/customers/notes',
  'api/customers/notes/',
  'api/notes',
  'api/notes/',
]) // Support multiple endpoints and trailing slashes
@UseGuards(JwtAuthGuard, PermissionsGuard)
export class CustomerNotesController {
  constructor(private readonly customerNotesService: CustomerNotesService) {}

 @Post()
async create(@Body() createCustomerNoteDto: CreateCustomerNoteDto, @User() user: any) {
  console.log('POST request received at', new Date().toISOString(), 'with payload:', JSON.stringify(createCustomerNoteDto, null, 2));
  try {
    const note = await this.customerNotesService.create(createCustomerNoteDto, user); // pass user here
    return {
      statusCode: HttpStatus.CREATED,
      message: 'Customer note created successfully',
      data: {
        id: note.id,
        note: note.note,
        customer_id: note.customer.id,
        parent_note_id: note.parentNote ? note.parentNote.id : null,
        created_by_id: note.createdBy ? note.createdBy.id : null, // this will be filled correctly now
        created_at: note.createdAt,
        updated_at: note.updatedAt,
      },
    };
  } catch (error) {
    console.error('POST error:', error.message, error.stack);
    throw error;
  }
}


  @Get()
  async findAll(
    @Query('customer', ParseIntPipe) customerId: number,
    @Query('ordering') ordering: string = '-created_at',
    @Query('limit', new DefaultValuePipe(10), ParseIntPipe) limit: number,
    @Query('offset', new DefaultValuePipe(0), ParseIntPipe) offset: number,
    @User() user: any,
  ) {
    console.log('GET request received at', new Date().toISOString(), 'with params:', { customerId, ordering, limit, offset });
    try {
      const { data, count, next, previous } = await this.customerNotesService.findAll(
        customerId,
        ordering,
        limit,
        offset,
        user,
      );
      return {
        next,
        previous,
        count,
        limit,
        offset,
        status: HttpStatus.OK,
        success: true,
        message: 'List Successfully Retrieved!',
        data,
      };
    } catch (error) {
      console.error('GET error:', error.message, error.stack);
      throw error;
    }
  }
}