import { Injectable, CanActivate, ExecutionContext, ForbiddenException, Logger } from '@nestjs/common';
import { Reflector } from '@nestjs/core';
import { UserRole } from '../../roles/roles.model';

@Injectable()
export class RolesGuard implements CanActivate {
  private readonly logger = new Logger(RolesGuard.name);

  constructor(private reflector: Reflector) {}

  canActivate(context: ExecutionContext): boolean {
    const requiredRoles = this.reflector.getAllAndOverride<UserRole[]>('roles', [
      context.getHandler(),
      context.getClass(),
    ]);

    if (!requiredRoles) {
      return true;
    }

    const { user } = context.switchToHttp().getRequest();
    
    // If user is Admin or Owner, allow access to everything
    if (user.role === 'Admin' || user.role === 'Owner') {
      this.logger.debug('Bypassing role check for Admin/Owner user');
      return true;
    }

    const hasRole = requiredRoles.some((role) => user.role === role);
    
    if (!hasRole) {
      this.logger.error(`User ${user.email} with role ${user.role} lacks required roles: ${requiredRoles.join(', ')}`);
      throw new ForbiddenException('You don\'t have the necessary permissions to access this resource');
    }
    
    return true;
  }
} 