import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { UserActivity } from './entities/user-activity.entity';
import { User } from '../users/entities/user.entity';
import { UserActivityListResponseDto, UserActivityResponseDto } from './dto/user-activity-response.dto';

@Injectable()
export class UserActivityService {
  constructor(
    @InjectRepository(UserActivity)
    private readonly activityRepo: Repository<UserActivity>,
  ) {}

  async logActivity(params: {
    user: User;
    action: string;
    type: string;
    ipAddress: string;
    changedObject: string;
    method: string;
  }): Promise<void> {
    const { user, action, type, ipAddress, changedObject, method } = params;

    const activity = this.activityRepo.create({
      username: user.email, // Assuming email is used for username
      action,
      type,
      ipAddress,
      changedObject,
      method,
      time: new Date(),
      user, // This sets the relation using @ManyToOne
    });

    await this.activityRepo.save(activity);
  }

  async getUserActivities(
    userId: number,
    limit: number = 10,
    offset: number = 0,
  ): Promise<UserActivityListResponseDto> {
    const [activities, total] = await this.activityRepo.findAndCount({
      where: { user: { id: userId } },
      relations: ['user'],
      order: { time: 'DESC' },
      take: limit,
      skip: offset,
    });

    const formattedActivities: UserActivityResponseDto[] = activities.map(activity => ({
      id: activity.id,
      user: {
        email: activity.user.email,
        name: `${activity.user.firstName} ${activity.user.lastName}`.trim(),
        last_login: activity.user.lastLoginAt?.toISOString() || null,
      },
      action: activity.action,
      method: activity.method,
      user_agent: activity.userAgent || '',
      ip_address: activity.ipAddress,
      changes: activity.changes || '',
      changed_object: activity.changedObject,
      time: activity.time.toISOString(),
      type: activity.type,
    }));

    return {
      previous: offset > 0 ? `/api/core/users_activity/?limit=${limit}&offset=${Math.max(0, offset - limit)}&user__id=${userId}` : null,
      count: total,
      limit,
      offset,
      status: 200,
      success: true,
      message: 'List Successfully Retrieved!',
      data: formattedActivities,
    };
  }
}
