import { Test, TestingModule } from '@nestjs/testing';
import { RolesController } from './roles.controller';
import { RolesService } from './roles.service';
import { getRepositoryToken } from '@nestjs/typeorm';
import { Role } from './entities/role.entity';
import { Repository } from 'typeorm';
import { NotFoundException } from '@nestjs/common';

describe('RolesController', () => {
  let controller: RolesController;
  let service: RolesService;

  const mockRolesService = {
    findAll: jest.fn(),
    findOne: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      controllers: [RolesController],
      providers: [
        {
          provide: RolesService,
          useValue: mockRolesService,
        },
      ],
    }).compile();

    controller = module.get<RolesController>(RolesController);
    service = module.get<RolesService>(RolesService);
  });

  it('should be defined', () => {
    expect(controller).toBeDefined();
  });

  describe('findAll', () => {
    it('should return an array of roles', async () => {
      const expectedRoles = [
        {
          id: 1,
          name: 'Test Role',
          description: 'Test Description',
          createdAt: new Date(),
          updatedAt: new Date(),
        },
      ];

      mockRolesService.findAll.mockResolvedValue(expectedRoles);

      const result = await controller.findAll();
      expect(result).toEqual(expectedRoles);
      expect(service.findAll).toHaveBeenCalled();
    });
  });

  describe('findOne', () => {
    it('should return a single role', async () => {
      const expectedRole = {
        id: 1,
        name: 'Test Role',
        description: 'Test Description',
        createdAt: new Date(),
        updatedAt: new Date(),
      };

      mockRolesService.findOne.mockResolvedValue(expectedRole);

      const result = await controller.findOne('1');
      expect(result).toEqual(expectedRole);
      expect(service.findOne).toHaveBeenCalledWith(1);
    });

    it('should throw NotFoundException if role not found', async () => {
      mockRolesService.findOne.mockRejectedValue(new NotFoundException('Role not found'));
      await expect(controller.findOne('999')).rejects.toThrow(NotFoundException);
    });
  });
}); 