import { Test, TestingModule } from '@nestjs/testing';
import { PermissionsController } from './permissions.controller';
import { PermissionsService } from './permissions.service';
import { UserRole } from '../roles/roles.model';
import { Permission } from './entities/permission.entity';

describe('PermissionsController', () => {
  let controller: PermissionsController;
  let service: PermissionsService;

  const mockPermissionsService = {
    findAll: jest.fn(),
    findOne: jest.fn(),
    findByCategory: jest.fn(),
    findByEntity: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      controllers: [PermissionsController],
      providers: [
        {
          provide: PermissionsService,
          useValue: mockPermissionsService,
        },
      ],
    }).compile();

    controller = module.get<PermissionsController>(PermissionsController);
    service = module.get<PermissionsService>(PermissionsService);
  });

  it('should be defined', () => {
    expect(controller).toBeDefined();
  });

  describe('findAll', () => {
    it('should return an array of permissions', async () => {
      const expectedPermissions = [
        {
          id: 1,
          category: 'users',
          entity: 'create',
          action: 'create_user',
          description: 'Create new user',
        },
      ];

      mockPermissionsService.findAll.mockResolvedValue(expectedPermissions);

      const result = await controller.findAll();
      expect(result).toEqual(expectedPermissions);
      expect(service.findAll).toHaveBeenCalled();
    });
  });

  describe('findByCategory', () => {
    it('should return permissions for a specific category', async () => {
      const category = 'users';
      const expectedPermissions = [
        {
          id: 1,
          category: 'users',
          entity: 'create',
          action: 'create_user',
          description: 'Create new user',
        },
      ];

      mockPermissionsService.findByCategory.mockResolvedValue(expectedPermissions);

      const result = await controller.findByCategory(category);
      expect(result).toEqual(expectedPermissions);
      expect(service.findByCategory).toHaveBeenCalledWith(category);
    });
  });

  describe('findByEntity', () => {
    it('should return permissions for a specific category and entity', async () => {
      const category = 'users';
      const entity = 'create';
      const expectedPermissions = [
        {
          id: 1,
          category: 'users',
          entity: 'create',
          action: 'create_user',
          description: 'Create new user',
        },
      ];

      mockPermissionsService.findByEntity.mockResolvedValue(expectedPermissions);

      const result = await controller.findByEntity(category, entity);
      expect(result).toEqual(expectedPermissions);
      expect(service.findByEntity).toHaveBeenCalledWith(category, entity);
    });
  });

  describe('findOne', () => {
    it('should return a single permission', async () => {
      const expectedPermission = {
        id: 1,
        category: 'users',
        entity: 'create',
        action: 'create_user',
        description: 'Create new user',
      };

      mockPermissionsService.findOne.mockResolvedValue(expectedPermission);

      const result = await controller.findOne('1');
      expect(result).toEqual(expectedPermission);
      expect(service.findOne).toHaveBeenCalledWith(1);
    });
  });
}); 