import { Injectable } from '@nestjs/common';
import { Request, Response } from 'express';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { User } from 'src/users/entities/user.entity';
import { UserActivityService } from 'src/user-activity/user-activity.service';

@Injectable()
export class LogoutService {
  constructor(
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    private readonly userActivityService: UserActivityService,
  ) {}

  async logout(req: Request, res: Response) {
    const sessionUser = req.session?.user;
    const jwtUser = req.user; // JWT payload set by JwtAuthGuard
    const sessionId = req.sessionID;

    // Check if there's a user in session or JWT
    if (!sessionUser && !jwtUser) {
      console.warn('No user found in session or JWT during logout');
      // If no session, clear cookie and return success to avoid errors
      res.clearCookie('connect.sid');
      return res.status(200).json({ message: 'Logout successful ' });
    }

    // Determine user email from session or JWT
    const userEmail = sessionUser?.email || jwtUser?.email;
    if (!userEmail) {
      console.warn('No valid user email found during logout');
      res.clearCookie('connect.sid');
      return res.status(200).json({ message: 'Logout successful (no valid user)' });
    }

    // Fetch user from database
    const user = await this.userRepo.findOne({ where: { email: userEmail } });
    if (!user) {
      console.warn('User not found in DB during logout:', userEmail);
      res.clearCookie('connect.sid');
      return res.status(200).json({ message: 'Logout successful (user not found)' });
    }

    // Log activity
    await this.userActivityService.logActivity({
      user,
      action: 'Logged Out',
      type: 'logout',
      ipAddress: req.ip || req.connection.remoteAddress || 'unknown',
      changedObject: `User ${user.email} logged out`,
      method: 'POST',
    });

    console.log('Logout activity logged. Destroying session:', sessionId);

    // Destroy session if it exists
    if (req.session) {
      req.session.destroy((err) => {
        if (err) {
          console.error('Failed to destroy session:', err);
          return res.status(500).json({ message: 'Logout failed' });
        }
        res.clearCookie('connect.sid');
        return res.status(200).json({ message: 'Logout successful' });
      });
    } else {
      // If no session, just clear cookie and return success
      res.clearCookie('connect.sid');
      return res.status(200).json({ message: 'Logout successful' });
    }
  }
}