import { Test, TestingModule } from '@nestjs/testing';
import { LoginController } from './login.controller';
import { LoginService } from './login.service';
import { Request, Response } from 'express';

interface SessionData {
  user?: {
    email: string;
    loginTime: string;
  };
}

describe('LoginController', () => {
  let controller: LoginController;
  let service: LoginService;

  const mockLoginService = {
    requestOTP: jest.fn(),
    verifyOTP: jest.fn(),
  };

  const mockRequest = {
    headers: {},
    session: {} as SessionData,
  } as Request;

  const mockResponse = {
    status: jest.fn().mockReturnThis(),
    json: jest.fn(),
  } as unknown as Response;

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      controllers: [LoginController],
      providers: [
        {
          provide: LoginService,
          useValue: mockLoginService,
        },
      ],
    }).compile();

    controller = module.get<LoginController>(LoginController);
    service = module.get<LoginService>(LoginService);
  });

  it('should be defined', () => {
    expect(controller).toBeDefined();
  });

  describe('generateOTP', () => {
    it('should generate OTP for valid email', async () => {
      const email = 'test@example.com';
      const expectedResponse = { message: 'OTP sent successfully' };

      mockLoginService.requestOTP.mockResolvedValue(expectedResponse);

      await controller.generateOTP(mockRequest, mockResponse, email);
      expect(mockResponse.status).toHaveBeenCalledWith(200);
      expect(mockResponse.json).toHaveBeenCalledWith(expectedResponse);
      expect(service.requestOTP).toHaveBeenCalledWith(email, mockRequest);
    });
  });

  describe('login', () => {
    it('should verify OTP and return user data with access token', async () => {
      const email = 'test@example.com';
      const otp = '123456';
      const mockUser = {
        id: 1,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'ADMIN',
        permissions: [],
      };

      const mockServiceResponse = {
        user: mockUser,
        accessToken: 'jwt-token',
        refreshToken: 'refresh-token',
      };

      const expectedResponse = {
        data: {
          name: 'John_Doe',
          email: 'test@example.com',
          role: 'ADMIN',
          tokens: {
            refresh: 'refresh-token',
            access: 'jwt-token',
          },
          permissions: [],
        },
        success: true,
        message: 'Logged in Successfully',
        status: 200,
      };

      mockLoginService.verifyOTP.mockResolvedValue(mockServiceResponse);

      await controller.login(mockRequest, mockResponse, email, otp);
      expect(mockResponse.status).toHaveBeenCalledWith(200);
      expect(mockResponse.json).toHaveBeenCalledWith(expectedResponse);
      expect(service.verifyOTP).toHaveBeenCalledWith(email, otp, mockRequest);
      expect(mockRequest.session.user).toBeDefined();
      expect(mockRequest.session.user?.email).toBe(email);
    });
  });
}); 