
import { Injectable, ConflictException, NotFoundException, UnauthorizedException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, In, Like } from 'typeorm';
import { Customer } from './entities/customer.entity';
import { Address } from './entities/address.entity';
import { CustomerNote } from './entities/customer-note.entity';
import { User } from '../users/entities/user.entity';
import { CreateCustomerDto } from './dto/create-customer.dto';
import { UpdateCustomerDto } from './dto/update-customer.dto';
import { ListCustomersDto } from './dto/list-customers.dto';
import { BulkAssignCustomerDto } from './dto/bulk-assign-customer.dto';

@Injectable()
export class CustomersService {
  constructor(
    @InjectRepository(Customer)
    private readonly customersRepository: Repository<Customer>,
    @InjectRepository(Address)
    private readonly addressesRepository: Repository<Address>,
    @InjectRepository(CustomerNote)
    private readonly customerNotesRepository: Repository<CustomerNote>,
    @InjectRepository(User)
    private readonly usersRepository: Repository<User>,
  ) {}

  async create(createCustomerDto: CreateCustomerDto, user: any): Promise<Customer> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const { email, first_name, last_name, home_phone, mobile_phone, address, title, iso, DNC, is_land_line_default, assigned_to_id } = createCustomerDto;

    if (email) {
      const existingCustomer = await this.customersRepository.findOne({ where: { email } });
      if (existingCustomer) {
        throw new ConflictException('Email already exists');
      }
    }

    if (assigned_to_id) {
      const assignee = await this.usersRepository.findOne({ where: { id: parseInt(assigned_to_id.toString(), 10) } });
      if (!assignee) {
        throw new NotFoundException(`User with ID ${assigned_to_id} not found`);
      }
    }

    const customer = this.customersRepository.create({
      firstName: first_name,
      lastName: last_name,
      email,
      mobileNumber: mobile_phone,
      landline: home_phone,
      title,
      iso,
      dnc: DNC,
      isLandLineDefault: is_land_line_default,
      assigned_to_id: assigned_to_id ? parseInt(assigned_to_id.toString(), 10) : user.id,
      createdById: user.id,
    });

    const savedCustomer = await this.customersRepository.save(customer);

    if (address && address.length > 0) {
      const addresses = address.map((addr) =>
        this.addressesRepository.create({
          ...addr,
          customer: savedCustomer,
          isActive: true,
        }),
      );
      await this.addressesRepository.save(addresses);
      savedCustomer.addresses = addresses;
    }

    return savedCustomer;
  }

  async update(id: number, updateCustomerDto: UpdateCustomerDto, user: any): Promise<Customer> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    console.log(`Updating customer with ID: ${id}`);

    const customer = await this.customersRepository.findOne({ where: { id }, relations: ['addresses'] });
    if (!customer) {
      throw new NotFoundException(`Customer with ID ${id} not found`);
    }

    const roleName = typeof user.role === 'string' ? user.role : user.role?.name || '';
    if (customer.assigned_to_id !== user.id && customer.createdById !== user.id && roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
      throw new UnauthorizedException('Not authorized to update this customer');
    }

    if (updateCustomerDto.email && updateCustomerDto.email !== customer.email) {
      const existingCustomer = await this.customersRepository.findOne({ where: { email: updateCustomerDto.email } });
      if (existingCustomer) {
        throw new ConflictException('Email already exists');
      }
    }

    if (updateCustomerDto.gen_customer_id && updateCustomerDto.gen_customer_id !== customer.genCustomerId) {
      const existingCustomer = await this.customersRepository.findOne({ where: { genCustomerId: updateCustomerDto.gen_customer_id } });
      if (existingCustomer) {
        throw new ConflictException('Generated customer ID already exists');
      }
    }

    if (updateCustomerDto.assigned_to_id) {
      const assignee = await this.usersRepository.findOne({ where: { id: parseInt(updateCustomerDto.assigned_to_id.toString(), 10) } });
      if (!assignee) {
        throw new NotFoundException(`User with ID ${updateCustomerDto.assigned_to_id} not found`);
      }
    }

    const updatedCustomer = this.customersRepository.merge(customer, {
      firstName: updateCustomerDto.first_name,
      lastName: updateCustomerDto.last_name,
      email: updateCustomerDto.email,
      mobileNumber: updateCustomerDto.mobile_phone,
      landline: updateCustomerDto.home_phone,
      title: updateCustomerDto.title,
      iso: updateCustomerDto.iso,
      dnc: updateCustomerDto.DNC,
      isLandLineDefault: updateCustomerDto.is_land_line_default,
      isActive: updateCustomerDto.is_active,
      customerId: updateCustomerDto.customer_id,
      genCustomerId: updateCustomerDto.gen_customer_id,
      assigned_to_id: updateCustomerDto.assigned_to_id ? parseInt(updateCustomerDto.assigned_to_id.toString(), 10) : customer.assigned_to_id,
    });

    const savedCustomer = await this.customersRepository.save(updatedCustomer);

    if (updateCustomerDto.address) {
      const existingAddresses = customer.addresses || [];
      const updatedAddressIds = updateCustomerDto.address
        .filter((addr) => addr.id)
        .map((addr) => addr.id);

      for (const existingAddress of existingAddresses) {
        if (!updatedAddressIds.includes(existingAddress.id)) {
          existingAddress.isActive = false;
          await this.addressesRepository.save(existingAddress);
        }
      }

      const addresses = await Promise.all(
        updateCustomerDto.address.map(async (addr) => {
          if (addr.id) {
            const existingAddress = existingAddresses.find((ea) => ea.id === addr.id);
            if (existingAddress) {
              return this.addressesRepository.merge(existingAddress, {
                ...addr,
                isActive: addr.is_active !== undefined ? addr.is_active : existingAddress.isActive,
              });
            }
          }
          return this.addressesRepository.create({
            ...addr,
            customer: savedCustomer,
            isActive: addr.is_active !== undefined ? addr.is_active : true,
          });
        }),
      );

      await this.addressesRepository.save(addresses);
      savedCustomer.addresses = addresses;
    }

    return savedCustomer;
  }

  async updateById(id: number, updateCustomerDto: UpdateCustomerDto, user: any): Promise<Customer> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    console.log(`Updating customer with id: ${id}`);

    const customer = await this.customersRepository.findOne({ where: { id }, relations: ['addresses'] });
    if (!customer) {
      throw new NotFoundException(`Customer with id ${id} not found`);
    }

    const roleName = typeof user.role === 'string' ? user.role : user.role?.name || '';
    if (customer.assigned_to_id !== user.id && customer.createdById !== user.id && roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
      throw new UnauthorizedException('Not authorized to update this customer');
    }

    if (updateCustomerDto.email && updateCustomerDto.email !== customer.email) {
      const existingCustomer = await this.customersRepository.findOne({ where: { email: updateCustomerDto.email } });
      if (existingCustomer) {
        throw new ConflictException('Email already exists');
      }
    }

    if (updateCustomerDto.gen_customer_id && updateCustomerDto.gen_customer_id !== customer.genCustomerId) {
      const existingCustomer = await this.customersRepository.findOne({ where: { genCustomerId: updateCustomerDto.gen_customer_id } });
      if (existingCustomer) {
        throw new ConflictException('Generated customer ID already exists');
      }
    }

    if (updateCustomerDto.assigned_to_id) {
      const assignee = await this.usersRepository.findOne({ where: { id: parseInt(updateCustomerDto.assigned_to_id.toString(), 10) } });
      if (!assignee) {
        throw new NotFoundException(`User with ID ${updateCustomerDto.assigned_to_id} not found`);
      }
    }

    const updatedCustomer = this.customersRepository.merge(customer, {
      firstName: updateCustomerDto.first_name,
      lastName: updateCustomerDto.last_name,
      email: updateCustomerDto.email,
      mobileNumber: updateCustomerDto.mobile_phone,
      landline: updateCustomerDto.home_phone,
      title: updateCustomerDto.title,
      iso: updateCustomerDto.iso,
      dnc: updateCustomerDto.DNC,
      isLandLineDefault: updateCustomerDto.is_land_line_default,
      isActive: updateCustomerDto.is_active,
      customerId: updateCustomerDto.customer_id,
      genCustomerId: updateCustomerDto.gen_customer_id,
      assigned_to_id: updateCustomerDto.assigned_to_id ? parseInt(updateCustomerDto.assigned_to_id.toString(), 10) : customer.assigned_to_id,
    });

    if (updateCustomerDto.address) {
      const existingAddresses = customer.addresses || [];
      const updatedAddressIds = updateCustomerDto.address
        .filter((addr) => addr.id)
        .map((addr) => addr.id);

      for (const existingAddress of existingAddresses) {
        if (!updatedAddressIds.includes(existingAddress.id)) {
          existingAddress.isActive = false;
          await this.addressesRepository.save(existingAddress);
        }
      }

      const addresses = await Promise.all(
        updateCustomerDto.address.map(async (addr) => {
          if (addr.id) {
            const existingAddress = existingAddresses.find((ea) => ea.id === addr.id);
            if (existingAddress) {
              return this.addressesRepository.merge(existingAddress, {
                ...addr,
                isActive: addr.is_active !== undefined ? addr.is_active : existingAddress.isActive,
              });
            }
          }
          return this.addressesRepository.create({
            ...addr,
            customer: updatedCustomer,
            isActive: addr.is_active !== undefined ? addr.is_active : true,
          });
        }),
      );
      updatedCustomer.addresses = addresses;
    }

    return this.customersRepository.save(updatedCustomer);
  }

  async findOne(id: number, user: any): Promise<any> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const customer = await this.customersRepository.findOne({
      where: { id },
      relations: ['addresses', 'assignedTo', 'assignedTo.reportsTo'],
    });
    if (!customer) {
      throw new NotFoundException(`Customer with ID ${id} not found`);
    }

    const roleName = typeof user.role === 'string' ? user.role : user.role || '';
    if (customer.assigned_to_id !== user.id && customer.createdById !== user.id && roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
      throw new UnauthorizedException('Not authorized to view this customer');
    }

    const notes = await this.customerNotesRepository.find({
      where: { customer: { id } },
      relations: ['parentNote'],
      order: { createdAt: 'DESC' },
    });

    const creator = await this.usersRepository.findOne({ where: { id: customer.createdById } });

    const transformedNotes = notes.map(note => ({
      id: note.id,
      replies: [],
      created_by: {
        id: user.id,
        user: {
          email: user.email,
          name: creator ? `${creator.firstName || ''} ${creator.lastName || ''}`.trim() || user.email : user.email,
          last_login: user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: user.createdAt || new Date(),
        updated_at: user.updatedAt || new Date(),
        is_active: user.isActive,
      },
      total_parents_count: 0,
      created_at: note.createdAt,
      updated_at: note.updatedAt,
      is_active: true,
      note: note.note,
      customer: customer.id,
      parent_note: note.parentNote ? note.parentNote.id : null,
      child_note: null,
    }));

    return {
      id: customer.id,
      customer_notes: transformedNotes,
      address: customer.addresses || [],
      assigned_to: customer.assignedTo ? {
        id: customer.assignedTo.id,
        user: {
          email: customer.assignedTo.email || '',
          name: `${customer.assignedTo.firstName || ''} ${customer.assignedTo.lastName || ''}`.trim() || customer.assignedTo.email || '',
          last_login: customer.assignedTo.lastLogin || null,
        },
        granted_permissions: customer.assignedTo.permissions || [],
        role: {
          id: '',
          name: customer.assignedTo.role || '',
          permissions: [],
        },
        reports_to: customer.assignedTo.reportsTo ? customer.assignedTo.reportsTo.id : null,
        created_at: customer.assignedTo.createdAt || null,
        updated_at: customer.assignedTo.updatedAt || null,
        is_active: customer.assignedTo.isActive ?? true,
      } : null,
      created_by: {
        id: user.id,
        user: {
          email: user.email,
          name: creator ? `${creator.firstName || ''} ${creator.lastName || ''}`.trim() || user.email : user.email,
          last_login: user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: user.createdAt || new Date(),
        updated_at: user.updatedAt || new Date(),
        is_active: user.isActive,
      },
      created_at: customer.createdAt,
      updated_at: customer.updatedAt,
      is_active: customer.isActive,
      customer_id: customer.customerId,
      gen_customer_id: customer.genCustomerId,
      title: customer.title,
      first_name: customer.firstName,
      last_name: customer.lastName,
      email: customer.email,
      mobile_phone: customer.dnc === true ? 'DNC' : customer.mobileNumber || '',
      home_phone: customer.dnc === true ? 'DNC' : customer.landline || '',
      is_land_line_default: customer.isLandLineDefault,
      iso: customer.iso,
      DNC: customer.dnc,
    };
  }

  async findOneByCustomerId(customerId: string, user: any): Promise<any> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    console.log(`Querying customer with customer_id: ${customerId} (type: ${typeof customerId}) for user ID: ${user.id}`);

    const customer = await this.customersRepository.findOne({
      where: { customerId: customerId.toString() },
      relations: ['addresses', 'assignedTo', 'assignedTo.reportsTo'],
    });

    if (!customer) {
      console.error(`No customer found with customer_id: ${customerId}`);
      throw new NotFoundException(`Customer with customer_id ${customerId} not found`);
    }

    const roleName = typeof user.role === 'string' ? user.role : user.role?.name || '';
    console.log(`User role: ${roleName}, assigned_to_id: ${customer.assigned_to_id}, createdById: ${customer.createdById}, user.id: ${user.id}`);
    if (customer.assigned_to_id !== user.id && customer.createdById !== user.id && roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
      console.error(`User ${user.id} not authorized to view customer ${customerId}`);
      throw new UnauthorizedException('Not authorized to view this customer');
    }

    const notes = await this.customerNotesRepository.find({
      where: { customer: { id: customer.id } },
      relations: ['parentNote'],
      order: { createdAt: 'DESC' },
    });

    const creator = await this.usersRepository.findOne({ where: { id: customer.createdById } });

    const transformedNotes = notes.map(note => ({
      id: note.id,
      replies: [],
      created_by: {
        id: user.id,
        user: {
          email: user.email,
          name: creator ? `${creator.firstName || ''} ${creator.lastName || ''}`.trim() || user.email : user.email,
          last_login: user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: user.createdAt || new Date(),
        updated_at: user.updatedAt || new Date(),
        is_active: user.isActive,
      },
      total_parents_count: 0,
      created_at: note.createdAt,
      updated_at: note.updatedAt,
      is_active: true,
      note: note.note,
      customer: customer.id,
      parent_note: note.parentNote ? note.parentNote.id : null,
      child_note: null,
    }));

    return {
      id: customer.id,
      customer_notes: transformedNotes,
      address: customer.addresses || [],
      assigned_to: customer.assignedTo ? {
        id: customer.assignedTo.id,
        user: {
          email: customer.assignedTo.email || '',
          name: `${customer.assignedTo.firstName || ''} ${customer.assignedTo.lastName || ''}`.trim() || customer.assignedTo.email || '',
          last_login: customer.assignedTo.lastLogin || null,
        },
        granted_permissions: customer.assignedTo.permissions || [],
        role: {
          id: '',
          name: customer.assignedTo.role || '',
          permissions: [],
        },
        reports_to: customer.assignedTo.reportsTo ? customer.assignedTo.reportsTo.id : null,
        created_at: customer.assignedTo.createdAt || null,
        updated_at: customer.assignedTo.updatedAt || null,
        is_active: customer.assignedTo.isActive ?? true,
      } : null,
      created_by: {
        id: user.id,
        user: {
          email: user.email,
          name: creator ? `${creator.firstName || ''} ${creator.lastName || ''}`.trim() || user.email : user.email,
          last_login: user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: user.createdAt || new Date(),
        updated_at: user.updatedAt || new Date(),
        is_active: user.isActive,
      },
      created_at: customer.createdAt,
      updated_at: customer.updatedAt,
      is_active: customer.isActive,
      customer_id: customer.customerId,
      gen_customer_id: customer.genCustomerId,
      title: customer.title,
      first_name: customer.firstName,
      last_name: customer.lastName,
      email: customer.email,
      mobile_phone: customer.dnc === true ? 'DNC' : customer.mobileNumber || '',
      home_phone: customer.dnc === true ? 'DNC' : customer.landline || '',
      is_land_line_default: customer.isLandLineDefault,
      iso: customer.iso,
      DNC: customer.dnc,
    };
  }

  async findAll(listCustomersDto: ListCustomersDto, user: any): Promise<{ data: any[]; total: number }> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const { offset = 0, limit = 10, search, scope } = listCustomersDto;
    const roleName = typeof user.role === 'string' ? user.role : user.role?.name || '';

    const qb = this.customersRepository.createQueryBuilder('customer')
      .leftJoinAndSelect('customer.addresses', 'addresses')
      .leftJoinAndSelect('customer.assignedTo', 'assignedTo')
      .leftJoinAndSelect('assignedTo.reportsTo', 'reportsTo')
      .skip(offset)
      .take(limit)
      .orderBy('customer.createdAt', 'DESC');

    if (roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
      qb.andWhere('(customer.assigned_to_id = :userId OR customer.createdById = :userId)', { userId: user.id });
    }

    const fieldMap: Record<string, string> = {
      first_name: 'customer.firstName',
      last_name: 'customer.lastName',
      email: 'customer.email',
      mobile_phone: 'customer.mobileNumber',
      home_phone: 'customer.landline',
      id: 'customer.id',
    };

    if (scope === 'created_by') {
      qb.andWhere('customer.createdById = :createdById', { createdById: user.id });
    } else if (scope && !fieldMap[scope]) {
      qb.andWhere('customer.iso = :iso', { iso: scope });
    }

    if (search) {
      if (scope && fieldMap[scope]) {
        qb.andWhere(`${fieldMap[scope]} LIKE :search`, { search: `%${search}%` });
      } else {
        qb.andWhere(
          [
            'customer.firstName LIKE :search',
            'customer.lastName LIKE :search',
            'customer.id LIKE :search',
            'customer.email LIKE :search',
            'customer.mobileNumber LIKE :search',
            'customer.landline LIKE :search',
          ].join(' OR '),
          { search: `%${search}%` }
        );
      }
    }

    const [data, total] = await qb.getManyAndCount();

    const creatorIds = [...new Set(data.map(customer => customer.createdById))];
    const creators = await this.usersRepository.find({ where: { id: In(creatorIds) } });
    const creatorMap = new Map(creators.map(creator => [creator.id, creator]));

    const transformedData = data.map(customer => ({
      id: customer.id,
      first_name: customer.firstName,
      last_name: customer.lastName,
      email: customer.email,
      mobile_phone: customer.dnc === true ? 'DNC' : customer.mobileNumber || '',
      home_phone: customer.dnc === true ? 'DNC' : customer.landline || '',
      address: customer.addresses || [],
      title: customer.title,
      iso: customer.iso,
      DNC: customer.dnc,
      is_land_line_default: customer.isLandLineDefault,
      created_at: customer.createdAt,
      customer_id: customer.customerId,
      gen_customer_id: customer.genCustomerId,
      assigned_to: customer.assignedTo ? {
        id: customer.assignedTo.id,
        user: {
          email: customer.assignedTo.email || '',
          name: `${customer.assignedTo.firstName || ''} ${customer.assignedTo.lastName || ''}`.trim() || customer.assignedTo.email || '',
          last_login: customer.assignedTo.lastLogin || null,
        },
        granted_permissions: customer.assignedTo.permissions || [],
        role: {
          id: '',
          name: customer.assignedTo.role || '',
          permissions: [],
        },
        reports_to: customer.assignedTo.reportsTo ? customer.assignedTo.reportsTo.id : null,
        created_at: customer.assignedTo.createdAt || null,
        updated_at: customer.assignedTo.updatedAt || null,
        is_active: customer.assignedTo.isActive ?? true,
      } : null,
      created_by: {
        id: customer.createdById,
        user: {
          email: creatorMap.get(customer.createdById)?.email || user.email,
          name: creatorMap.get(customer.createdById) ? 
            `${creatorMap.get(customer.createdById)?.firstName || ''} ${creatorMap.get(customer.createdById)?.lastName || ''}`.trim() || user.email : user.email,
          last_login: creatorMap.get(customer.createdById)?.lastLogin || user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: creatorMap.get(customer.createdById)?.createdAt || user.createdAt || new Date(),
        updated_at: creatorMap.get(customer.createdById)?.updatedAt || user.updatedAt || new Date(),
        is_active: creatorMap.get(customer.createdById)?.isActive ?? user.isActive,
      },
    }));

    return { data: transformedData, total };
  }

  async bulkAssign(dto: BulkAssignCustomerDto, user: any): Promise<string> {
    const { customer_ids, assigned_to_id } = dto;

    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const assignee = await this.usersRepository.findOne({ where: { id: assigned_to_id } });
    if (!assignee) {
      throw new NotFoundException(`User with ID ${assigned_to_id} not found`);
    }

    const customers = await this.customersRepository.findBy({ id: In(customer_ids) });

    if (!customers.length) {
      throw new NotFoundException(`No customers found with the provided IDs`);
    }

    for (const customer of customers) {
      customer.assigned_to_id = assigned_to_id;
    }

    await this.customersRepository.save(customers);

    return `Selected customers have been assigned to ${assignee.firstName || ''}_${assignee.lastName || 'user'}`;
  }
}