import { Controller, Post, Get, Patch, Body, Query, HttpStatus, Param, UseGuards, BadRequestException } from '@nestjs/common';
import { CustomersService } from './customers.service';
import { CreateCustomerDto } from './dto/create-customer.dto';
import { UpdateCustomerDto } from './dto/update-customer.dto';
import { ListCustomersDto } from './dto/list-customers.dto';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { PermissionsGuard } from '../auth/guards/permissions.guard';
import { RequirePermission } from '../auth/decorators/require-permission.decorator';
import { User } from '../auth/decorators/user.decorator';
import { BulkAssignCustomerDto } from './dto/bulk-assign-customer.dto';

@Controller('api/customer/management')
@UseGuards(JwtAuthGuard, PermissionsGuard)
export class CustomersController {
  constructor(private readonly customersService: CustomersService) {}

  @Post()
  @RequirePermission('Can add customer')
  async create(@Body() createCustomerDto: CreateCustomerDto, @User() user: any) {
    const customer = await this.customersService.create(createCustomerDto, user);
    return {
      statusCode: HttpStatus.CREATED,
      message: 'Customer profile created successfully',
      data: {
        iso: customer.iso,
        title: customer.title,
        assigned_to_id: customer.assigned_to_id,
        last_name: customer.lastName,
        first_name: customer.firstName,
        email: customer.email,
        home_phone: customer.landline,
        mobile_phone: customer.mobileNumber,
        id: customer.id,
        gen_customer_id: customer.genCustomerId,
      },
    };
  }

 
  @Get(':id')
  @RequirePermission('Can view customer')
  async findOne(@Param('id') id: number, @User() user: any) {
    try {
      const customer = await this.customersService.findOne(id, user);
      return {
        statusCode: HttpStatus.OK,
        message: 'Customer retrieved successfully',
        data: customer,
      };
    } catch (error) {
      console.error(`Error retrieving customer with id ${id} for user ${user?.id || 'unknown'}:`, error.message, error.stack);
      throw error;
    }
  }

  @Patch('bulk_assign')
  @RequirePermission('Can change customer')
  async bulkAssignCustomers(@Body() dto: BulkAssignCustomerDto, @User() user: any) {
    const result = await this.customersService.bulkAssign(dto, user);
    return {
      statusCode: 200,
      success: true,
      message: {
        detail: 'Selected customers have been assigned',
      },
      data: result,
    };
  }

  @Patch(':id')
  @RequirePermission('Can change customer')
  async update(@Param('id') id: string, @Body() updateCustomerDto: UpdateCustomerDto, @User() user: any) {
    try {
      const customer = await this.customersService.updateById(Number(id), updateCustomerDto, user);
      return {
        statusCode: HttpStatus.OK,
        message: 'Customer profile updated successfully',
        data: {
          iso: customer.iso,
          title: customer.title,
          assigned_to_id: customer.assigned_to_id,
          last_name: customer.lastName,
          first_name: customer.firstName,
          email: customer.email,
          home_phone: customer.landline,
          mobile_phone: customer.mobileNumber,
          id: customer.id,
          gen_customer_id: customer.genCustomerId,
        },
      };
    } catch (error) {
      console.error(`Error updating customer with id ${id}:`, error.message);
      throw error;
    }
  }

  @Get()
  @RequirePermission('Can view customer')
  async findAll(@Query() listCustomersDto: ListCustomersDto, @User() user: any) {
    const { data, total } = await this.customersService.findAll(listCustomersDto, user);
    return {
      statusCode: HttpStatus.OK,
      message: 'Customers retrieved successfully',
      data,
      total,
    };
  }
}