import { Test, TestingModule } from '@nestjs/testing';
import { CustomersController } from './customers.controller';
import { CustomersService } from './customers.service';
import { CreateCustomerDto } from './dto/create-customer.dto';
import { ListCustomersDto } from './dto/list-customers.dto';
import { HttpStatus } from '@nestjs/common';

describe('CustomersController', () => {
  let controller: CustomersController;
  let service: CustomersService;

  const mockCustomer = {
    id: 1,
    firstName: 'Ramesh',
    lastName: 'Takoba',
    email: 'rohitznaik78924@gmail.com',
    mobileNumber: '9945566683',
    landline: '07892269418',
    address: [],
    title: 'Ms',
    iso: 'GB-ENG',
    dnc: false,
    isLandLineDefault: true,
    createdAt: new Date(),
    assigned_to_id: 1,
    genCustomerId: undefined,
  };

  const mockTransformedCustomer = {
    id: 1,
    first_name: 'Ramesh',
    last_name: 'Takoba',
    email: 'rohitznaik78924@gmail.com',
    mobile_phone: '9945566683',
    home_phone: '07892269418',
    title: 'Ms',
    iso: 'GB-ENG',
    assigned_to_id: 1,
    gen_customer_id: undefined,
  };

  const mockCustomersService = {
    create: jest.fn().mockResolvedValue(mockCustomer),
    findAll: jest.fn().mockResolvedValue({ data: [mockTransformedCustomer], total: 1 }),
    findOne: jest.fn().mockResolvedValue(mockCustomer),
    updateById: jest.fn().mockResolvedValue(mockCustomer),
    bulkAssign: jest.fn().mockResolvedValue('Selected customers have been assigned to John_Doe'),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      controllers: [CustomersController],
      providers: [
        {
          provide: CustomersService,
          useValue: mockCustomersService,
        },
      ],
    }).compile();

    controller = module.get<CustomersController>(CustomersController);
    service = module.get<CustomersService>(CustomersService);
  });

  it('should be defined', () => {
    expect(controller).toBeDefined();
  });

  describe('create', () => {
    it('should create a customer and return success response', async () => {
      const dto: CreateCustomerDto = {
        first_name: 'Ramesh',
        last_name: 'Takoba',
        email: 'rohitznaik78924@gmail.com',
        mobile_phone: '9945566683',
        home_phone: '07892269418',
        address: [],
        title: 'Ms',
        iso: 'GB-ENG',
        DNC: false,
        is_land_line_default: true,
        assigned_to_id: 1,
      };
      const user = { id: 1, role: 'admin' };
      const result = await controller.create(dto, user);
      expect(result).toEqual({
        statusCode: HttpStatus.CREATED,
        message: 'Customer profile created successfully',
        data: {
          iso: 'GB-ENG',
          title: 'Ms',
          assigned_to_id: 1,
          last_name: 'Takoba',
          first_name: 'Ramesh',
          email: 'rohitznaik78924@gmail.com',
          home_phone: '07892269418',
          mobile_phone: '9945566683',
          id: 1,
          gen_customer_id: undefined,
        },
      });
      expect(service.create).toHaveBeenCalledWith(dto, user);
    });
  });

  describe('findAll', () => {
    it('should return a list of customers', async () => {
      const listCustomersDto: ListCustomersDto = {
        offset: 0,
        limit: 10,
        search: '',
        scope: '',
      };
      const user = { id: 1, role: 'admin' };
      const result = await controller.findAll(listCustomersDto, user);
      expect(result).toEqual({
        statusCode: HttpStatus.OK,
        message: 'Customers retrieved successfully',
        data: [mockTransformedCustomer],
        total: 1,
      });
      expect(service.findAll).toHaveBeenCalledWith(listCustomersDto, user);
    });
  });

  describe('findOne', () => {
    it('should return a customer by ID', async () => {
      const mockCustomer = { id: 1, name: 'Test Customer' };
      service.findOne = jest.fn().mockResolvedValue(mockCustomer);
      const user = { id: 1, role: 'admin' };
      const result = await controller.findOne(1, user);
      expect(result).toEqual({
        statusCode: HttpStatus.OK,
        message: 'Customer retrieved successfully',
        data: mockCustomer,
      });
      expect(service.findOne).toHaveBeenCalledWith(1, user);
    });
  });

  describe('update', () => {
    it('should update a customer and return updated data', async () => {
      const mockUpdatedCustomer = { id: 1, firstName: 'Updated', lastName: 'Customer', email: 'updated@example.com', landline: '', mobileNumber: '', title: '', iso: '', assigned_to_id: 2, genCustomerId: 'GEN123' };
      service.updateById = jest.fn().mockResolvedValue(mockUpdatedCustomer);
      const user = { id: 1, role: 'admin' };
      const dto = {
        first_name: 'Updated',
        last_name: 'Customer',
        email: 'updated@example.com',
        mobile_phone: '',
        home_phone: '',
        address: [],
        title: '',
        iso: '',
        DNC: false,
        is_land_line_default: false,
        is_active: true,
        customer_id: '',
        gen_customer_id: 'GEN123',
        assigned_to_id: 2,
      };
      const result = await controller.update('1', dto, user);
      expect(result).toEqual({
        statusCode: HttpStatus.OK,
        message: 'Customer profile updated successfully',
        data: {
          iso: '',
          title: '',
          assigned_to_id: 2,
          last_name: 'Customer',
          first_name: 'Updated',
          email: 'updated@example.com',
          home_phone: '',
          mobile_phone: '',
          id: 1,
          gen_customer_id: 'GEN123',
        },
      });
      expect(service.updateById).toHaveBeenCalledWith(1, dto, user);
    });
  });

  describe('bulkAssignCustomers', () => {
    it('should bulk assign customers and return a success message', async () => {
      const dto = { customer_ids: [1, 2], assigned_to_id: 3 };
      const user = { id: 1, role: 'admin' };
      const mockResult = 'Selected customers have been assigned to John_Doe';
      service.bulkAssign = jest.fn().mockResolvedValue(mockResult);
      const result = await controller.bulkAssignCustomers(dto, user);
      expect(result).toEqual({
        statusCode: 200,
        success: true,
        message: { detail: 'Selected customers have been assigned' },
        data: mockResult,
      });
      expect(service.bulkAssign).toHaveBeenCalledWith(dto, user);
    });
  });
});