import { Injectable, Logger } from '@nestjs/common';
import * as nodemailer from 'nodemailer';
import { ConfigService } from '@nestjs/config';

@Injectable()
export class EmailService {
  private readonly logger = new Logger(EmailService.name);
  private transporter: nodemailer.Transporter;

  constructor(private configService: ConfigService) {
    const port = this.configService.get('SMTP_PORT');
    const secure = port === '465'; // Only set secure to true for port 465

    this.transporter = nodemailer.createTransport({
      host: this.configService.get('SMTP_HOST'),
      port: parseInt(port, 10),
      secure: secure,
      auth: {
        user: this.configService.get('SMTP_USER'),
        pass: this.configService.get('SMTP_PASS'),
      },
      tls: {
        // Do not fail on invalid certs
        rejectUnauthorized: false
      }
    });

    // Verify transporter configuration
    this.transporter.verify((error, success) => {
      if (error) {
        this.logger.error('SMTP Connection Error:', error);
      } else {
        this.logger.log('SMTP Server is ready to send messages');
      }
    });
  }

  async sendLoginInstructions(email: string, password: string): Promise<void> {
    try {
      await this.transporter.sendMail({
        from: this.configService.get('SMTP_FROM'),
        to: email,
        subject: 'Welcome to CRM System - Login Instructions',
        html: `
          <h1>Welcome to CRM System</h1>
          <p>Your account has been created successfully.</p>
          <p>Here are your login credentials:</p>
          <ul>
            <li>Email: ${email}</li>
            <li>Password: ${password}</li>
          </ul>
          <p>Please change your password after first login.</p>
          <p>Best regards,<br>Natures Healthcare</p>
        `,
      });
      this.logger.log(`Login instructions sent to ${email}`);
    } catch (error) {
      this.logger.error(`Failed to send email to ${email}:`, error);
      throw error;
    }
  }

  async sendOTP(email: string, otp: string): Promise<void> {
    try {
      await this.transporter.sendMail({
        from: this.configService.get('SMTP_FROM'),
        to: email,
        subject: 'Your OTP for Login',
        html: `
          <div style="text-align:center;">
            <h1 style="font-size:2.2em;font-weight:bold;margin-bottom:0.5em;">Your OTP for Login</h1>
            <p>Dear User,</p>
            <p>To complete your login process, please use the following One-Time Password (OTP). This OTP is valid for the next 10 minutes.</p>
            <div style="display:inline-block;margin:32px 0;">
              <span style="display:inline-block;background:#1677ff;color:#fff;font-size:2em;font-weight:bold;padding:18px 48px;border-radius:8px;letter-spacing:4px;">${otp}</span>
            </div>
          </div>
        `,
      });
      this.logger.log(`OTP sent to ${email}`);
    } catch (error) {
      this.logger.error(`Failed to send OTP email to ${email}:`, error);
      throw error;
    }
  }

  async sendWelcomeEmail(email: string): Promise<void> {
    try {
      await this.transporter.sendMail({
        from: this.configService.get('SMTP_FROM'),
        to: email,
        subject: 'Welcome to CRM System - Login Instructions',
        html: `
          <h1>Welcome to CRM System</h1>
          <p>Your account has been created successfully.</p>
          <p>To log in to the system:</p>
          <ol>
            <li>Go to the login page</li>
            <li>Enter your email address: ${email}</li>
            <li>Click on "Request OTP"</li>
            <li>You will receive an OTP via email</li>
            <li>Enter the OTP to complete your login</li>
          </ol>
          <p>For security reasons, the OTP will expire in 5 minutes.</p>
          <p>If you have any issues logging in, please contact your system administrator.</p>
          <p>Best regards,<br>Natures Healthcare</p>
        `,
      });
      this.logger.log(`Login instructions sent to ${email}`);
    } catch (error) {
      this.logger.error(`Failed to send login instructions to ${email}:`, error);
      throw error;
    }
  }

  async sendWelcomeEmailWithButton(email: string, name: string): Promise<void> {
    const appUrl = this.configService.get('APP_URL') || 'https://crm.natureshealthcare.co.uk';
    try {
      await this.transporter.sendMail({
        from: this.configService.get('SMTP_FROM'),
        to: email,
        subject: "Welcome to Natures Healthcare!",
        html: `
          <div style="text-align:center;">
            <h1>Welcome to Natures Healthcare!</h1>
            <p>Dear ${name},</p>
            <p>We are excited to have you on board and look forward to working with you. We believe that you will be a great addition to our team, bringing your unique skills and perspectives.</p>
            <p>If you have any questions or need any assistance, please feel free to reach out to us. We're here to help you get settled and succeed in your new role.</p>
            <a href="${appUrl}" style="display:inline-block;padding:12px 32px;background:#43b05c;color:#fff;text-decoration:none;border-radius:6px;font-size:18px;margin:20px 0;">Click here to get started</a>
            <p>Thank you,<br>Nature's Healthcare</p>
          </div>
        `,
      });
      this.logger.log(`Welcome email with button sent to ${email}`);
    } catch (error) {
      this.logger.error(`Failed to send welcome email with button to ${email}:`, error);
      throw error;
    }
  }

  async sendReportsToNotification(
    reportsToEmail: string,
    reportsToName: string,
    employeeName: string,
    employeeEmail: string
  ): Promise<void> {
    try {
      await this.transporter.sendMail({
        from: this.configService.get('SMTP_FROM'),
        to: reportsToEmail,
        subject: "New Team Member Assignment",
        html: `
          <div style="text-align:center;">
            <h1>New Team Member Assignment</h1>
            <p>Dear ${reportsToName},</p>
            <p>You have been assigned as the supervisor for a new team member:</p>
            <div style="background:#f5f5f5;padding:20px;margin:20px 0;border-radius:8px;">
              <p style="margin:0;"><strong>Employee Name:</strong> ${employeeName}</p>
              <p style="margin:10px 0;"><strong>Email:</strong> ${employeeEmail}</p>
            </div>
            <p>As their supervisor, you will be responsible for:</p>
            <ul style="text-align:left;display:inline-block;margin:20px 0;">
              <li>Providing guidance and support</li>
              <li>Reviewing their work and performance</li>
              <li>Helping them integrate into the team</li>
              <li>Addressing any concerns or questions they may have</li>
            </ul>
            <p>Please reach out to them to introduce yourself and establish a working relationship.</p>
            <p>Thank you for your leadership and support!</p>
            <p>Best regards,<br>Natures Healthcare</p>
          </div>
        `,
      });
      this.logger.log(`Reports to notification sent to ${reportsToEmail}`);
    } catch (error) {
      this.logger.error(`Failed to send reports to notification to ${reportsToEmail}:`, error);
      throw error;
    }
  }
} 