"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var JwtTokenActivityService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JwtTokenActivityService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const jwt_token_activity_entity_1 = require("../entities/jwt-token-activity.entity");
let JwtTokenActivityService = JwtTokenActivityService_1 = class JwtTokenActivityService {
    jwtTokenActivityRepository;
    logger = new common_1.Logger(JwtTokenActivityService_1.name);
    constructor(jwtTokenActivityRepository) {
        this.jwtTokenActivityRepository = jwtTokenActivityRepository;
    }
    logTokenExpiration(token, userId, duration) {
        const expirationMessage = `
╔════════════════════════════════════════════════════════════════════════════╗
║                           TOKEN EXPIRATION ALERT                           ║
╠════════════════════════════════════════════════════════════════════════════╣
║ Token: ${token.substring(0, 20)}...                                        ║
║ User ID: ${userId}                                                         ║
║ Duration Used: ${duration} minutes                                         ║
║ Expired At: ${new Date().toISOString()}                                   ║
╚════════════════════════════════════════════════════════════════════════════╝
`;
        console.log(expirationMessage);
        this.logger.warn(`Token expired for user ${userId} after ${duration} minutes of use`);
    }
    async logTokenActivity(token, userId, req) {
        try {
            const expiresAt = new Date();
            expiresAt.setMinutes(expiresAt.getMinutes() + 15);
            const tokenActivity = this.jwtTokenActivityRepository.create({
                token,
                user_id: userId,
                expiresAt,
                ipAddress: req.ip,
                userAgent: req.headers['user-agent'],
                lastUsedAt: new Date(),
            });
            await this.jwtTokenActivityRepository.save(tokenActivity);
            const creationMessage = `
╔════════════════════════════════════════════════════════════════════════════╗
║                           NEW TOKEN CREATED                                ║
╠════════════════════════════════════════════════════════════════════════════╣
║ Token: ${token.substring(0, 20)}...                                        ║
║ User ID: ${userId}                                                         ║
║ Created At: ${new Date().toISOString()}                                   ║
║ Expires At: ${expiresAt.toISOString()}                                    ║
╚════════════════════════════════════════════════════════════════════════════╝
`;
            console.log(creationMessage);
            this.logger.debug(`Token activity logged for user ${userId}. Expires at: ${expiresAt.toISOString()}`);
        }
        catch (error) {
            this.logger.error(`Failed to log token activity: ${error.message}`);
        }
    }
    async updateTokenLastUsed(token) {
        try {
            const tokenActivity = await this.jwtTokenActivityRepository.findOne({
                where: { token, isValid: true }
            });
            if (!tokenActivity) {
                this.logger.warn(`Token not found or invalid: ${token}`);
                return;
            }
            const now = new Date();
            const timeUntilExpiry = tokenActivity.expiresAt.getTime() - now.getTime();
            const minutesUntilExpiry = Math.floor(timeUntilExpiry / (1000 * 60));
            if (timeUntilExpiry <= 0) {
                const duration = now.getTime() - tokenActivity.createdAt.getTime();
                const minutesUsed = Math.floor(duration / (1000 * 60));
                this.logTokenExpiration(token, tokenActivity.user_id, minutesUsed);
                await this.markTokenAsExpired(token);
                return;
            }
            const usageMessage = `
╔════════════════════════════════════════════════════════════════════════════╗
║                           TOKEN USAGE UPDATE                               ║
╠════════════════════════════════════════════════════════════════════════════╣
║ Token: ${token.substring(0, 20)}...                                        ║
║ User ID: ${tokenActivity.user_id}                                          ║
║ Minutes Until Expiry: ${minutesUntilExpiry}                                ║
║ Last Used At: ${now.toISOString()}                                        ║
╚════════════════════════════════════════════════════════════════════════════╝
`;
            console.log(usageMessage);
            await this.jwtTokenActivityRepository.update({ token, isValid: true }, { lastUsedAt: now });
        }
        catch (error) {
            this.logger.error(`Failed to update token last used: ${error.message}`);
        }
    }
    async invalidateToken(token) {
        try {
            const tokenActivity = await this.jwtTokenActivityRepository.findOne({
                where: { token }
            });
            if (!tokenActivity) {
                this.logger.warn(`Token not found for invalidation: ${token}`);
                return;
            }
            const duration = new Date().getTime() - tokenActivity.createdAt.getTime();
            const minutesUsed = Math.floor(duration / (1000 * 60));
            const invalidationMessage = `
╔════════════════════════════════════════════════════════════════════════════╗
║                           TOKEN INVALIDATED                                ║
╠════════════════════════════════════════════════════════════════════════════╣
║ Token: ${token.substring(0, 20)}...                                        ║
║ User ID: ${tokenActivity.user_id}                                          ║
║ Duration Used: ${minutesUsed} minutes                                      ║
║ Invalidated At: ${new Date().toISOString()}                               ║
╚════════════════════════════════════════════════════════════════════════════╝
`;
            console.log(invalidationMessage);
            await this.jwtTokenActivityRepository.update({ token }, { isValid: false });
        }
        catch (error) {
            this.logger.error(`Failed to invalidate token: ${error.message}`);
        }
    }
    async markTokenAsExpired(token) {
        try {
            const tokenActivity = await this.jwtTokenActivityRepository.findOne({
                where: { token }
            });
            if (!tokenActivity) {
                this.logger.warn(`Token not found for expiration: ${token}`);
                return;
            }
            const duration = new Date().getTime() - tokenActivity.createdAt.getTime();
            const minutesUsed = Math.floor(duration / (1000 * 60));
            this.logTokenExpiration(token, tokenActivity.user_id, minutesUsed);
            await this.jwtTokenActivityRepository.update({ token }, { isExpired: true, isValid: false });
        }
        catch (error) {
            this.logger.error(`Failed to mark token as expired: ${error.message}`);
        }
    }
    async cleanupExpiredTokens() {
        try {
            const tenDaysAgo = new Date();
            tenDaysAgo.setDate(tenDaysAgo.getDate() - 10);
            const result = await this.jwtTokenActivityRepository.delete({
                createdAt: (0, typeorm_2.LessThan)(tenDaysAgo)
            });
            this.logger.debug(`Cleaned up ${result.affected} expired tokens older than 10 days`);
        }
        catch (error) {
            this.logger.error(`Failed to cleanup expired tokens: ${error.message}`);
        }
    }
    async getTokenActivities(userId, limit = 10, offset = 0) {
        try {
            const [activities, total] = await this.jwtTokenActivityRepository.findAndCount({
                where: { user_id: userId },
                order: { createdAt: 'DESC' },
                take: limit,
                skip: offset,
            });
            const activitiesWithDuration = activities.map(activity => {
                const duration = activity.lastUsedAt
                    ? activity.lastUsedAt.getTime() - activity.createdAt.getTime()
                    : 0;
                const minutesUsed = Math.floor(duration / (1000 * 60));
                return {
                    ...activity,
                    duration_minutes: minutesUsed,
                    time_until_expiry: activity.expiresAt.getTime() - new Date().getTime(),
                };
            });
            this.logger.debug(`Retrieved ${activities.length} token activities for user ${userId}`);
            return {
                data: activitiesWithDuration,
                total,
                limit,
                offset,
            };
        }
        catch (error) {
            this.logger.error(`Failed to get token activities: ${error.message}`);
            throw error;
        }
    }
};
exports.JwtTokenActivityService = JwtTokenActivityService;
exports.JwtTokenActivityService = JwtTokenActivityService = JwtTokenActivityService_1 = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(jwt_token_activity_entity_1.JwtTokenActivity)),
    __metadata("design:paramtypes", [typeorm_2.Repository])
], JwtTokenActivityService);
//# sourceMappingURL=jwt-token-activity.service.js.map